package com.xebialabs.xlrelease.api.internal;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;
import com.google.common.base.Objects;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.TaskApi;
import com.xebialabs.xlrelease.api.v1.views.TaskAccessView;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.repository.query.TaskBasicData;
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionLogService;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.TaskGranularPermissions;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.views.*;
import com.xebialabs.xlrelease.views.converters.CommentViewConverter;
import com.xebialabs.xlrelease.views.converters.TasksViewConverter;
import com.xebialabs.xlrelease.views.converters.UserViewConverter;
import com.xebialabs.xlrelease.views.tasks.TaskListRelease;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.notifications.actors.extension.TaskWatcherExecutionActorMessages.AddWatcher;
import static com.xebialabs.xlrelease.notifications.actors.extension.TaskWatcherExecutionActorMessages.RemoveWatcher;
import static com.xebialabs.xlrelease.notifications.actors.extension.TaskWatcherExecutionActorMessages.UpdateWatchers;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static jakarta.ws.rs.core.Response.Status.OK;
import static jakarta.ws.rs.core.Response.status;
import static java.util.Collections.emptyList;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

/**
 * A task in a release.
 */
@Path("/tasks")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class TaskResource {

    private TaskApi taskApi;

    private ReleaseService releaseService;

    private TaskService taskService;

    private PermissionChecker permissions;

    private TaskAccessService taskAccessService;

    private TaskSearchService taskSearchService;

    private ReleaseActorService releaseActorService;

    private TasksViewConverter tasksViewConverter;

    private UserViewConverter userViewConverter;

    private CommentViewConverter commentViewConverter;

    private TaskGranularPermissions taskPermissionChecker;
    private TaskExecutionLogService taskExecutionLogService;
    private SseService sseService;

    @Autowired
    public TaskResource(TaskApi taskApi,
                        ReleaseService releaseService,
                        TaskService taskService,
                        PermissionChecker permissions,
                        TaskAccessService taskAccessService,
                        TaskSearchService taskSearchService,
                        ReleaseActorService releaseActorService,
                        TasksViewConverter tasksViewConverter,
                        UserViewConverter userViewConverter,
                        CommentViewConverter commentViewConverter,
                        TaskGranularPermissions taskPermissionChecker,
                        TaskExecutionLogService taskExecutionLogService,
                        SseService sseService) {
        this.taskApi = taskApi;
        this.releaseService = releaseService;
        this.taskService = taskService;
        this.permissions = permissions;
        this.taskAccessService = taskAccessService;
        this.taskSearchService = taskSearchService;
        this.releaseActorService = releaseActorService;
        this.tasksViewConverter = tasksViewConverter;
        this.userViewConverter = userViewConverter;
        this.commentViewConverter = commentViewConverter;
        this.taskPermissionChecker = taskPermissionChecker;
        this.taskExecutionLogService = taskExecutionLogService;
        this.sseService = sseService;
    }

    /**
     * Adds a new task to a container (phase or task group).
     *
     * @param containerId the identifier of the container
     * @param taskForm    the information of the new task
     * @return the new task
     */
    @POST
    @Timed
    @Path("{containerId:.*(Phase|Task)[^/-]*}")
    public TaskFullView addTask(@PathParam("containerId") @IdParam String containerId, TaskForm taskForm) {
        String releaseId = releaseIdFrom(containerId);
        var pc = permissions.context();
        pc.checkView(releaseId);
        pc.checkEdit(releaseId);
        pc.checkEditTask(releaseId);

        Task task = releaseActorService.createTask(containerId, taskForm.toTask());
        return tasksViewConverter.toFullView(task, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    /**
     * Returns the details of a task.
     *
     * @param taskId the identifier of the task
     * @return the task
     */
    @GET
    @Timed
    @Path("{taskId:[^/]*Task[^/-]*}")
    public TaskFullView getTask(@PathParam("taskId") @IdParam String taskId) {
        Task task = taskService.findByIdIncludingArchived(taskId);
        permissions.checkViewTask(task);
        return tasksViewConverter.toFullView(task, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    /**
     * Returns the list of tasks accessible to the current user, grouped by release.
     *
     * @param tasksFilters   the TasksFilters criteria
     * @param page            next page to query
     * @param numberByPage    releases per page
     * @return the matching tasks, grouped by release.
     */
    @POST
    @Timed
    @Path("search")
    public Page<TaskListRelease> getTasksByRelease(@QueryParam("page") Integer page,
                                                   @QueryParam("numberbypage") Integer numberByPage,
                                                   TasksFilters tasksFilters) {
        int defaultLimit = TaskSearchService.DEFAULT_TASK_LIMIT();
        checkArgument(numberByPage <= defaultLimit, "Number of results per page cannot be more than %d", defaultLimit);
        return taskSearchService.getTasksByRelease(tasksFilters, (page != null) ? page : 0, ofNullable(numberByPage).orElse(defaultLimit));
    }

    /**
     * Completes a running task manually.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/complete")
    public Response completeTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.completeTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Completes tasks manually.
     *
     * @param commentTasksView the comment view to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("complete")
    public BulkActionResultView completeTasks(CommentTasksView commentTasksView) {
        List<String> filteredTaskIds = permissions.filterTasksWithTaskTransitionPermission(commentTasksView.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(releaseActorService.completeTasks(filteredTaskIds, commentTasksView.getCommentText()));
    }

    /**
     * Skips a manual task.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/skip")
    public Response skipTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.skipTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Skips tasks.
     *
     * @param comment the comment to associate to the action
     */
    @POST
    @Timed
    @Path("skip")
    public BulkActionResultView skipTasks(CommentTasksView comment) {
        checkNotNull(comment.getCommentText(), "Comment is mandatory when skipping tasks.");
        List<String> filteredTaskIds = permissions.filterTasksWithTaskTransitionPermission(comment.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(releaseActorService.skipTasks(filteredTaskIds, comment.getCommentText(), AUTHENTICATED_USER));
    }

    /**
     * Fails a task.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/fail")
    public Response failTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.failTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Fails tasks.
     *
     * @param comment a comment describing the 'fail' action.
     */
    @POST
    @Timed
    @Path("fail")
    public BulkActionResultView failTasks(CommentTasksView comment) {
        checkNotNull(comment.getCommentText(), "Comment is mandatory when failing tasks.");
        List<String> filteredTaskIds = permissions.filterTasksWithTaskTransitionPermission(comment.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(releaseActorService.failTasksManually(filteredTaskIds, comment.getCommentText()));
    }

    /**
     * Aborts a task.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/abort")
    public Response abortTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.abortTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Retries a task.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/retry")
    public Response retryTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.retryTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Retries tasks.
     *
     * @param comment the comment to associate to the action
     */
    @POST
    @Timed
    @Path("retry")
    public BulkActionResultView retryTasks(CommentTasksView comment) {
        checkNotNull(comment.getCommentText(), "Comment is mandatory when retrying tasks.");
        List<String> filteredTaskIds = permissions.filterTasksWithTaskTransitionPermission(comment.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(releaseActorService.retryTasks(comment.getTaskIds(), comment.getCommentText()));
    }

    /**
     * Aborts tasks.
     *
     * @param comment the comment to associate to the action
     */
    @POST
    @Timed
    @Path("abort")
    public BulkActionResultView abortTasks(CommentTasksView comment) {
        checkNotNull(comment.getCommentText(), "Comment is mandatory when aborting tasks.");
        List<String> filteredTaskIds = permissions.filterTasksWithTaskTransitionPermission(comment.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(releaseActorService.abortTasks(comment.getTaskIds(), comment.getCommentText()));
    }

    /**
     * Forces a pending task to start immediately.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/startNow")
    public Response startNow(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.start(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Reopens a task that had been completed in advance.
     *
     * @param taskId  the identifier of the task
     * @param comment the comment to associate to the action
     * @return HTTP status 200 to indicate success
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/reopen")
    public Response reopenTask(@PathParam("taskId") @IdParam String taskId, CommentView comment) {
        taskApi.reopenTask(taskId, fromView(comment));
        return status(OK).build();
    }

    /**
     * Reopens tasks.
     *
     * @param comment the comment to associate to the action
     */
    @POST
    @Timed
    @Path("reopen")
    public BulkActionResultView reopenTasks(CommentTasksView comment) {
        checkNotNull(!isNullOrEmpty(comment.getCommentText()), "Comment is mandatory when reopening tasks.");
        permissions.checkReopenTasksInRelease(comment.getTaskIds());
        if (comment.getTaskIds().isEmpty()) {
            return new BulkActionResultView(emptyList());
        }
        return new BulkActionResultView(releaseActorService.reopenTasks(comment.getTaskIds(), comment.getCommentText()));
    }

    /**
     * Returns the comments associated with a task.
     *
     * @param taskId the identifier of the task
     * @return the comments
     */
    @GET
    @Timed
    @Path("{taskId:.*Task[^/-]*}/comments")
    public List<CommentView> getCommentsOfTask(@PathParam("taskId") @IdParam String taskId) {
        return taskService.getCommentsOfTask(taskId).stream().map(commentViewConverter::toFullView).collect(toList());
    }

    /**
     * Adds a comment to a task.
     *
     * @param taskId      the identifier of the task
     * @param commentView the comment's data
     * @return the comment
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/comments")
    public CommentView addComment(@PathParam("taskId") @IdParam String taskId, CommentView commentView) {
        permissions.checkIsAllowedToCommentOnTask(taskId);
        Comment comment = taskService.addComment(taskId, commentView.getText());
        return commentViewConverter.toFullView(comment);
    }

    @POST
    @Timed
    @Path("comments")
    public BulkActionResultView addCommentToTasks(CommentTasksView commentTasksView) {
        List<String> filteredTaskIds = permissions.filterAllowedToCommentOnTasks(commentTasksView.getTaskIds());
        if (filteredTaskIds.isEmpty()) {
            return new BulkActionResultView(filteredTaskIds);
        }
        return new BulkActionResultView(taskService.addComments(filteredTaskIds, commentTasksView.getCommentText()));
    }

    /**
     * Updates a task and notifies the new owner if the task has been reassigned.
     *
     * @param taskId      the identifier of the task
     * @param updatedTask the new task values
     * @return the updated task
     */
    @PUT
    @Timed
    @Path("{taskId:.*Task[^/-]*}")
    public TaskFullView updateTask(@PathParam("taskId") @IdParam String taskId, TaskFullView updatedTask) {
        Set<TaskUpdateDirective> updateDirectives = taskPermissionChecker.getUpdateDirectives(releaseIdFrom(taskId));
        Task result = releaseActorService.updateTask(taskId, tasksViewConverter.toTask(updatedTask), updateDirectives);

        return tasksViewConverter.toFullView(result, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    /**
     * Change task type to the target task type.
     *
     * @param taskId     the identifier of the task
     * @param targetType the new task type
     * @return new task
     */
    @POST
    @Timed
    @Path("{taskId:.*Task[^/-]*}/changeType")
    public TaskFullView changeTaskType(@PathParam("taskId") @IdParam String taskId, @QueryParam("targetType") String targetType) {
        final Task result = taskApi.changeTaskType(taskId, targetType);
        return tasksViewConverter.toFullView(result, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    @PUT
    @Timed
    @Path("reassign")
    public BulkActionResultView reassignTasks(ReassignTasksView reassignTasks) {
        permissions.checkReassignTasks(reassignTasks.getTaskIds(), reassignTasks.getOwner());
        if (reassignTasks.getTaskIds().isEmpty()) {
            return new BulkActionResultView(emptyList());
        }
        return new BulkActionResultView(releaseActorService.reassignTasks(reassignTasks.getTaskIds(), reassignTasks.getTeam(), reassignTasks.getOwner()));
    }

    /**
     * Reassigns a task to a different owner.
     *
     * @param taskId      the identifier of the task
     * @param updatedTask the details of the updated task, including the new owner
     */
    @PUT
    @Timed
    @Path("{taskId:.*Task[^/-]*}/owner")
    public void reassignToOwner(@PathParam("taskId") @IdParam String taskId, TaskFullView updatedTask) {
        String newOwner = updatedTask != null ? updatedTask.getOwnerUsername() : null;
        permissions.checkReassignTaskToUser(taskId, newOwner);
        releaseActorService.reassignTaskToOwner(taskId, newOwner);
    }

    /**
     * Deletes a task owner.
     *
     * @param taskId the identifier of the task
     */
    @DELETE
    @Timed
    @Path("{taskId:.*Task[^/-]*}/owner")
    public void removeOwner(@PathParam("taskId") @IdParam String taskId) {
        reassignToOwner(taskId, null);
    }

    /**
     * Reassigns a task to a different team.
     *
     * @param taskId      the identifier of the task
     * @param updatedTask the details of the updated task, including the new team
     */
    @PUT
    @Timed
    @Path("{taskId:.*Task[^/-]*}/team")
    public void reassignToTeam(@PathParam("taskId") @IdParam String taskId, TaskFullView updatedTask) {
        Task task = taskService.findById(taskId);
        permissions.checkReassignTaskPermission(task.getRelease());
        String previousTeam = task.getTeam();
        String newTeam = updatedTask != null ? updatedTask.getTeam() : null;
        if (!Objects.equal(newTeam, previousTeam)) {
            releaseActorService.reassignTaskToTeam(taskId, newTeam);
        }
    }

    /**
     * Removes the team of a task.
     *
     * @param taskId the identifier of the task
     */
    @DELETE
    @Timed
    @Path("{taskId:.*Task[^/-]*}/team")
    public void removeTeam(@PathParam("taskId") @IdParam String taskId) {
        reassignToTeam(taskId, null);
    }

    @DELETE
    public BulkActionResultView deleteTasks(List<String> taskIds) {
        permissions.checkDeleteTasks(taskIds);
        if (taskIds.isEmpty()) {
            return new BulkActionResultView(emptyList());
        }
        return new BulkActionResultView(releaseActorService.deleteTasks(taskIds));
    }

    /**
     * Deletes a task.
     *
     * @param taskId the identifier of the task
     */
    @DELETE
    @Timed
    @Path("{taskId:.*Task[^/-]*}")
    public void deleteTask(@PathParam("taskId") @IdParam String taskId) {
        taskApi.delete(taskId);
    }

    /**
     * Returns a list of tasks.
     *
     * @param form the identifiers of the tasks
     * @return the tasks
     */
    @POST
    @Timed
    @Path("poll")
    public List<TaskPollingView> poll(DomainIdsForm form) {
        List<TaskBasicData> taskList = taskService.findTasksForPolling(form.getIds());
        return taskList.stream().map(taskData -> {
            TaskPollingView view = new TaskPollingView();
            view.setId(taskData.taskId());
            view.setStatus(taskData.status());
            view.setStatusLine(taskData.statusLine());
            return view;
        }).collect(toList());
    }

    /**
     * Returns the custom task definitions created through Digital.ai Release's extensibility mechanism.
     *
     * @return the definitions
     */
    @GET
    @Timed
    @Path("task-definitions")
    public Collection<TaskDefinition> getTaskDefinitions() {
        return taskAccessService.getTaskDefinitions();
    }

    /**
     * Returns the properties used to configure a custom task type.
     *
     * @param scriptDefinitionType the task type
     * @return the definition of the type, which includes the properties
     */
    @GET
    @Timed
    @Path("task-definitions/{scriptDefinitionType}")
    public TaskWithPropertiesDefinition getCustomScriptProperties(@PathParam("scriptDefinitionType") String scriptDefinitionType) {
        return new TaskWithPropertiesDefinition(Type.valueOf(scriptDefinitionType));
    }

    /**
     * Returns all task accesses.
     * It's necessary to have the 'security#edit' permission to call this end point.
     *
     * @return the task accesses
     */
    @GET
    @Timed
    @Path("types-access")
    public List<TaskAccessView> getTaskAccesses() {
        permissions.check(EDIT_SECURITY);
        return taskAccessService.getTaskAccesses();
    }

    /**
     * Update task accesses.
     * It's necessary to have the 'security#edit' permission to call this end point.
     *
     * @param taskAccesses the task accesses to save
     */
    @PUT
    @Timed
    @Path("types-access")
    public void updateTaskAccesses(List<TaskAccessView> taskAccesses) {
        permissions.check(EDIT_SECURITY);
        taskAccessService.updateTaskAccesses(taskAccesses);
    }

    /**
     * Returns a release view for given task.
     *
     * @param taskId the task identifier
     * @return task release view
     */
    @GET
    @Timed
    @Path("{taskId:.*Task[^/-]*}/release")
    public TaskReleaseView getTaskRelease(@PathParam("taskId") @IdParam String taskId) {
        Task task = taskService.findByIdIncludingArchived(taskId);
        permissions.checkViewTask(task);
        Release release = releaseService.findByIdIncludingArchived(releaseIdFrom(taskId));
        return new TaskReleaseView(task, release);
    }

    @GET
    @Path("tags")
    public Set<String> getAllTags() {
        return taskService.getAllTags(500);
    }

    private com.xebialabs.xlrelease.api.v1.forms.Comment fromView(final CommentView comment) {
        return new com.xebialabs.xlrelease.api.v1.forms.Comment(comment.getText());
    }

    @Path("{taskId:.*Task[^/-]*}/watchers")
    @POST
    public Set<UserView> addWatcher(@PathParam("taskId") @IdParam String taskId, UserView watcherView) {
        if (watcherView.getUsername() == null || watcherView.getUsername().trim().isEmpty()) {
            throw new IllegalArgumentException("Username should not be null or empty");
        }
        if (!AUTHENTICATED_USER.getName().equals(watcherView.getUsername())) {
            permissions.checkEditTask(releaseIdFrom(taskId));
        }
        Set<String> watchers = releaseActorService.executeCommand(taskId, new AddWatcher(taskId, watcherView.getUsername()));
        return convertToView(watchers);
    }

    @Path("{taskId:.*Task[^/-]*}/watchers/{username}")
    @DELETE
    public Set<UserView> removeWatcher(@PathParam("taskId") @IdParam String taskId, @PathParam("username") String username) {
        if (!AUTHENTICATED_USER.getName().equals(username)) {
            permissions.checkEditTask(releaseIdFrom(taskId));
        }
        Set<String> watchers = releaseActorService.executeCommand(taskId, new RemoveWatcher(taskId, username));
        return convertToView(watchers);
    }

    @Path("{taskId:.*Task[^/-]*}/watchers")
    @PUT
    public Set<UserView> updateWatchers(@PathParam("taskId") @IdParam String taskId, Set<UserView> watchersView) {
        permissions.checkEditTask(releaseIdFrom(taskId));
        Set<String> watchers = watchersView.stream().map(UserView::getUsername).collect(Collectors.toSet());
        Set<String> updatedWatchers = releaseActorService.executeCommand(taskId, new UpdateWatchers(taskId, watchers));
        return convertToView(updatedWatchers);
    }

    private Set<UserView> convertToView(Set<String> watchers) {
        return watchers.stream().map(username -> userViewConverter.toUserView(username))
                .collect(Collectors.toSet());
    }


    /**
     * Streams entries for a given task execution.
     * <p>
     * IMPORTANT: streamed events MAY BE DUPLICATED if fromStart is true
     *
     * @param taskId task id of the task
     * @param executionId execution id of the task (This is not the same as the custom script task execution id)
     */
    @POST
    @Path("{taskId:.*Task[^/-]*}/execution/{executionId}/log/follow")
    public void followLog(@PathParam("taskId") @IdParam String taskId,
                          @PathParam("executionId") String executionId) {
        Task task = taskService.findByIdIncludingArchived(taskId, ResolveOptions.WITHOUT_DECORATORS());
        permissions.checkViewTask(task);
        taskExecutionLogService.watch(taskId, executionId);
    }

    @DELETE
    @Path("{taskId:.*Task[^/-]*}/execution/{executionId}/log/follow")
    public void unfollowLog(@PathParam("taskId") @IdParam String taskId,
                            @PathParam("executionId") String executionId) {
        taskExecutionLogService.stopWatch(executionId);
    }


    @GET
    @Path("{taskId:.*Task[^/-]*}/execution/{executionId}/log")
    @Produces(MediaType.TEXT_PLAIN)
    public Response log(@PathParam("taskId") @IdParam String taskId,
                        @PathParam("executionId") String executionId,
                        @QueryParam("job") @DefaultValue("9223372036854775807") Long job,
                        @QueryParam("chunk") @DefaultValue("9223372036854775807") Long chunk) {
        Task task = taskService.findByIdIncludingArchived(taskId, ResolveOptions.WITHOUT_DECORATORS());
        permissions.checkViewTask(task);
        return ResponseHelper.streamFile(
                "task-execution-" + executionId + ".log",
                output -> taskExecutionLogService.fetch(taskId, executionId, output, job, chunk),
                MediaType.TEXT_PLAIN
        );
    }

    /**
     * Retrieve list of task executions. It's a starting point of execution flows to display task execution logs.
     * <p>
     * execution flow 1:
     * <ul>
     * <li> UI hits `GET getTaskExecutions` and in response gets all task executions
     * <li> UI gets back information: (id - execution id, executionNo - order, modified date, end date, latest_job_id, latest_chunk)
     * <li> UI decides to ask `GET followLog` to stream log entries - and it will get event with current status: (status, latest_job_id, latest_chunk_id)
     * <ul>
     *  <li> status can be: 'in_progress', 'closed' or 'unknown'
     *  <li> if status is 'unknown' latest_job_id is -1 and latest_chunk is -1</li>
     * </ul>
     * <li> IN PARALLEL: UI invokes `GET log` endpoint with 'latest_job_id' and 'latest_chunk' parameters and fetches content up until that point
     * (100s of streamed megabytes) - if needed backend can also discard everything but latest 40KB of data (or whatever we choose)
     * <li> UI decides how to present this data AND any incoming events to the user
     * </ul>
     * <p>
     * execution flow 2:
     * <ul>
     * <li> UI hits `GET getTaskExecutions` and in response gets all task executions
     * <li> UI gets back information: (id - execution id, executionNo - order, modified date, end date, latest_job_id, latest_chunk)
     * <li> UI decides to ask `GET followLog` to stream log entries - and it will get event with current status: (status, latest_job_id, latest_chunk_id)
     * <ul>
     *  <li> status can be: 'in_progress', 'closed' or 'unknown'
     *  <li> if status is 'unknown' latest_job_id is -1 and latest_chunk is -1</li>
     * </ul>
     * </ul>
     *
     * @param taskId task id
     * @return list of task execution data
     */
    @GET
    @Path("{taskId:.*Task[^/-]*}/executions")
    public List<TaskExecutionLogView> getTaskExecutions(@PathParam("taskId") @IdParam String taskId) {
        Task task = taskService.findByIdIncludingArchived(taskId, ResolveOptions.WITHOUT_DECORATORS());
        permissions.checkViewTask(task);
        return taskExecutionLogService.fetchAllExecutions(taskId);
    }

    @POST
    @Path("{taskId:.*Task[^/-]*}/follow")
    public void followTask(@PathParam("taskId") @IdParam String taskId) {
        Task task = taskService.findByIdIncludingArchived(taskId, ResolveOptions.WITHOUT_DECORATORS());
        permissions.checkViewTask(task);
        sseService.subscribeTopicToUser(task.getId());
    }

    @DELETE
    @Path("{taskId:.*Task[^/-]*}/follow")
    public void unfollowTask(@PathParam("taskId") @IdParam String taskId) {
        Task task = taskService.findByIdIncludingArchived(taskId, ResolveOptions.WITHOUT_DECORATORS());
        sseService.unsubscribeTopicToUser(task.getId());
    }

}
