package com.xebialabs.xlrelease.domain.tasks.gate;

import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Dependency;
import com.xebialabs.xlrelease.domain.GateTask;
import com.xebialabs.xlrelease.domain.tasks.TaskExecutor;
import com.xebialabs.xlrelease.repository.DependencyRepository;
import com.xebialabs.xlrelease.repository.DependencyTargetResolver;
import com.xebialabs.xlrelease.script.TaskSoftReference;
import com.xebialabs.xlrelease.user.User;

import scala.Option;

import static com.xebialabs.xlrelease.domain.FailureReasons.GATE_TASK_DEPENDS_ON_AN_ABORTED_RELEASE;
import static com.xebialabs.xlrelease.domain.FailureReasons.GATE_TASK_FAILED;
import static com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED;
import static java.lang.String.format;
import static scala.jdk.javaapi.CollectionConverters.asScala;


@Component
public class GateTaskExecutor implements TaskExecutor<GateTask> {
    private static final Logger logger = LoggerFactory.getLogger(GateTaskExecutor.class);
    private DependencyRepository dependencyRepository;
    private ReleaseActorService releaseActorService;
    private DependencyTargetResolver dependencyTargetResolver;

    @Autowired
    public GateTaskExecutor(DependencyRepository dependencyRepository,
                            ReleaseActorService releaseActorService,
                            DependencyTargetResolver dependencyTargetResolver) {
        this.dependencyRepository = dependencyRepository;
        this.releaseActorService = releaseActorService;
        this.dependencyTargetResolver = dependencyTargetResolver;
    }

    @Override
    public Class<GateTask> getTaskClass() {
        return GateTask.class;
    }

    @Override
    public void execute(TaskSoftReference<GateTask> taskRef) {
        GateTask task = taskRef.get();
        try {
            logger.trace("starting execute for {}", task.getId());
            tryResolveTargetIdOfDependencies(task);

            completeOrFailIfNeeded(task);
        } catch (TargetIdResolutionFailedException e) {
            releaseActorService.failTaskAsync(task.getId(), GATE_TASK_FAILED.format(e.getMessage()), User.AUTHENTICATED_USER, Option.empty());
        }
    }

    private void tryResolveTargetIdOfDependencies(GateTask gateTask) throws TargetIdResolutionFailedException {
        List<Dependency> updatedDependencies = new ArrayList<>();

        for (Dependency dependency : gateTask.getDependencies()) {
            String targetId = dependency.getTargetId();
            if (Strings.isEmpty(targetId)) {
                if (!dependency.hasResolvedTarget() && !dependency.isArchived()) {
                    throw new TargetIdResolutionFailedException(format("Missing target on dependency %s. " +
                            "Was it defined as a variable with an empty value?", dependency.getId()));
                }
                continue;
            }
            dependencyTargetResolver.resolveTarget(dependency);
            logger.trace("tryResolveTargetIdOfDependencies {} resolved target {}, isArchived = {}, isDone = {}", gateTask.getId(), dependency.getTargetId(), dependency.isArchived(), dependency.isDone());
            updatedDependencies.add(dependency);
        }
        dependencyRepository.update(gateTask.getRelease(), asScala(updatedDependencies).toSeq());
    }

    private void completeOrFailIfNeeded(final GateTask task) {
        if (task.isCompletable()) {
            releaseActorService.markTaskAsDoneAsync(COMPLETED, task.getId(), null, User.AUTHENTICATED_USER);
        } else if (task.hasAbortedDependencies()) {
            String titles = task.getAbortedDependencyTitles();
            releaseActorService.failTaskAsync(task.getId(), GATE_TASK_DEPENDS_ON_AN_ABORTED_RELEASE.format(titles), User.AUTHENTICATED_USER, Option.empty());
        }
    }

}
