package com.xebialabs.xlrelease.api.internal;

import java.util.Date;
import java.util.List;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import jakarta.ws.rs.core.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.calendar.ReleaseCalendar;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.calendar.Blackout;
import com.xebialabs.xlrelease.domain.calendar.SpecialDay;
import com.xebialabs.xlrelease.param.DateParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.CalendarService;
import com.xebialabs.xlrelease.service.ReleaseService;

import static jakarta.ws.rs.core.Response.Status.NOT_FOUND;

/**
 * The release calendar.
 */
@Path("/calendar")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class CalendarResource {

    private ReleaseService releaseService;

    private CalendarService calendarService;

    private PermissionChecker permissions;


    @Autowired
    public CalendarResource(ReleaseService releaseService, CalendarService calendarService, PermissionChecker permissions) {
        this.releaseService = releaseService;
        this.calendarService = calendarService;
        this.permissions = permissions;
    }

    @GET
    @Timed
    @Path("/subscribe/{calendarToken}")
    public Response subscribe(@PathParam("calendarToken") String calendarToken) {
        Release release = releaseService.findByCalendarToken(calendarToken);
        if (release == null || !release.isCalendarPublished()) {
            return Response.status(NOT_FOUND).build();
        }

        String ics = new ReleaseCalendar(release).toString();

        return Response.ok(ics).build();
    }

    @GET
    @Timed
    @Path("/specialDays/{specialDayId:.*/Calendar/[0-9^/]{8}}")
    public SpecialDay getSpecialDay(@PathParam("specialDayId") String specialDayId) {
        return calendarService.getSpecialDay(specialDayId);
    }

    @PUT
    @Timed
    @Path("/specialDays/{specialDayId:.*/Calendar/[0-9^/]{8}}")
    public SpecialDay setSpecialDay(@PathParam("specialDayId") String specialDayId, SpecialDay specialDay) {
        return calendarService.setSpecialDay(specialDayId, specialDay);
    }

    @GET
    @Timed
    @Path("/specialDays")
    public List<SpecialDay> getSpecialDays(@QueryParam("from") @DateParam Date from, @QueryParam("to") @DateParam Date to) {
        return calendarService.getSpecialDays(from, to);
    }

    @GET
    @Timed
    @Path("/blackouts")
    public List<Blackout> getBlackouts(@QueryParam("from") @DateParam Date from,
                                       @QueryParam("to") @DateParam Date to) {
        return calendarService.getBlackouts(from, to);
    }

    @GET
    @Timed
    @Path("/blackouts/{blackoutId:.*/Blackout[^/]*}")
    public Blackout getBlackout(@PathParam("blackoutId") String blackoutId) {
        return calendarService.getBlackout(blackoutId);
    }

    @POST
    @Timed
    @Path("/blackouts")
    public Blackout addBlackout(Blackout blackout) {
        permissions.check(XLReleasePermissions.EDIT_GLOBAL_BLACKOUT);
        return calendarService.addBlackout(blackout);
    }

    @PUT
    @Timed
    @Path("/blackouts/{blackoutId:.*/Blackout[^/]*}")
    public Blackout updateBlackout(@PathParam("blackoutId") String blackoutId, Blackout blackout) {
        permissions.check(XLReleasePermissions.EDIT_GLOBAL_BLACKOUT);
        blackout.setId(blackoutId);
        return calendarService.updateBlackout(blackout);
    }

    @DELETE
    @Timed
    @Path("/blackouts/{blackoutId:.*/Blackout[^/]*}")
    public void deleteBlackout(@PathParam("blackoutId") String blackoutId) {
        permissions.check(XLReleasePermissions.EDIT_GLOBAL_BLACKOUT);
        calendarService.deleteBlackout(blackoutId);
    }

    @GET
    @Timed
    @Path("/blackouts/withinBlackout")
    public boolean withinBlackout() {
        return calendarService.isInBlackout(new Date());
    }
}
