package com.xebialabs.xlrelease.actors

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.actors.ReleaseActor.Activate
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.AttachmentMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.DependenciesMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ExecutionServiceMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.GateMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.PhasesMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ReleaseMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TaskMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TeamMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TemplateMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.VariableMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages._
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.{ReleaseBulkAbortedEvent, ReleaseBulkEvent, ReleaseBulkStartedEvent, ReleaseExecutionEvent}
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.events.{Subscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.PhaseVersion
import com.xebialabs.xlrelease.scheduler.workers.Worker.CreateReleaseTaskExecutionResult
import com.xebialabs.xlrelease.script.ContainerTaskResult
import com.xebialabs.xlrelease.script.DefaultScriptService.{BaseScriptTaskResults, CustomScriptTaskResults, ScriptTaskResults}
import com.xebialabs.xlrelease.service.ExecutorServiceUtils
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.views.MovementIndexes
import grizzled.slf4j.Logging
import org.apache.pekko.actor.ActorRef
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout
import org.apache.pekko.util.Timeout._
import org.springframework.beans.factory.DisposableBean
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import java.util.concurrent.{CompletableFuture, ScheduledExecutorService, TimeUnit}
import java.util.{Date, List => JUList}
import scala.annotation.varargs
import scala.concurrent.{Await, Future, TimeoutException}
import scala.jdk.CollectionConverters._
import scala.jdk.FutureConverters._
import scala.util.{Failure, Success, Try}


/**
 * This service proxies all calls to release* actors and encapsulates all internals related to 'ask' operation and futures handling.
 *
 * Java code should access release* actors via this service.
 */
// scalastyle:off number.of.methods
@Component("releaseActorService")
class ReleaseActorService @Autowired()(
                                        config: XlrConfig,
                                        releasesActorHolder: ReleasesActorHolder,
                                        eventBus: XLReleaseEventBus
                                      ) extends Logging with DisposableBean {
  def skipTaskDueToPreconditionCheck(taskId: String, executionId: String, executionLog: String): Unit = askAndAwaitWithRetry(taskId) {
    SkipTaskDueToPreconditionCheck(taskId, executionId, executionLog)
  }

  val timeoutExecutor: ScheduledExecutorService = config.executors.timeoutExecutor.pool

  lazy val releaseActorRef: ActorRef = releasesActorHolder.awaitActorRef()

  eventBus.register(this)

  private def askAndAwait[T](id: String)(msg: AnyRef): T = {
    implicit val askTimeout: Timeout = config.timeouts.releaseActionResponse
    logger.debug(s"askAndAwait ${id} ${msg.toString}")
    val result = try {
      Await.result(
        releaseActorRef ? ReleaseAction(releaseIdFrom(id), msg),
        askTimeout.duration
      ).asInstanceOf[T]
    } catch {
      case e: TimeoutException =>
        logger.error(s"Got TimeoutException in askAndAwait ${id} ${msg.toString}", e)
        throw new RuntimeException(e)
      case e: InterruptedException =>
        logger.error(s"Got RuntimeException in askAndAwait ${id} ${msg.toString}", e)
        throw new RuntimeException(e)
      case e: Throwable =>
        logger.error(s"Got exception in askAndAwait ${id} ${msg.toString}", e)
        throw e
    }
    logger.debug(s"askAndAwait resolved ${id} ${msg.toString}")
    result
  }

  private def askAndAwaitWithRetry[T](id: String)(msg: AnyRef): T = {
    val future = new CompletableFuture[Any]()
    val askTimeout: Timeout = config.timeouts.messageRetryTimeout
    askFutureRetryLoop(releaseIdFrom(id), msg, future, askTimeout)
    Await.result(
      future.asScala,
      askTimeout.duration.mul(config.timeouts.messageRetryAttempts)
    ).asInstanceOf[T]
  }

  private def askMultipleAndAwait[T >: Null <: AnyRef](ids: Seq[String])(msg: Seq[String] => AnyRef): Seq[T] = {
    implicit val askTimeout: Timeout = config.timeouts.releaseActionResponse
    try {
      ids.groupBy(releaseIdFrom)
        .map { case (releaseId, groupedIds) => releaseActorRef ? ReleaseAction(releaseId, msg(groupedIds)) }
        .map { future =>
          Try(Await.result(future, askTimeout.duration).asInstanceOf[T]).getOrElse(null)
        }.filter(_ != null).toSeq
    } catch {
      case e: TimeoutException => throw new RuntimeException(e)
      case e: InterruptedException => throw new RuntimeException(e)
    }
  }

  private def tell(id: String)(msg: AnyRef): Unit = {
    logger.debug(s"Forwarding release action '$msg' to '$releaseActorRef'")
    releaseActorRef ! ReleaseAction(releaseIdFrom(id), msg)
  }

  def activate(id: String): Unit = tell(id)(Activate)

  //Script Service
  def saveScriptResults(taskId: String, scriptTaskResults: ScriptTaskResults): Unit = askAndAwait(taskId) {
    ScriptServiceMessages.SaveScriptResults(taskId, scriptTaskResults)
  }

  def saveCustomScriptResults(taskId: String, results: CustomScriptTaskResults, executionId: String): Unit = askAndAwaitWithRetry(taskId) {
    ScriptServiceMessages.SaveCustomScriptResults(taskId, results, executionId)
  }

  def startRelease(releaseId: String, user: User, releaseStartedImmediatelyAfterBeingCreated: Boolean = false): Release = askAndAwait(releaseId) {
    ExecutionServiceMessages.StartRelease(user, releaseStartedImmediatelyAfterBeingCreated)
  }

  def startRelease(releaseId: String, user: User): Release = askAndAwaitWithRetry(releaseId) {
    ExecutionServiceMessages.StartRelease(user)
  }

  def startReleaseWithoutResponse(releaseId: String, user: User): Unit = askAndAwaitWithRetry(releaseId) {
    ExecutionServiceMessages.StartRelease(user, needsResponse = false)
  }

  def startReleases(releaseIds: JUList[String], user: User): JUList[Release] = {
    val releases = askMultipleAndAwait[Release](releaseIds.asScala.toSeq) { _ =>
      ExecutionServiceMessages.StartRelease(user, isPartOfBulkOperation = true)
    }.asJava
    if (!releases.isEmpty()) {
      publishBulkOperationEvent(ReleaseBulkStartedEvent(releases))
    }
    releases
  }

  private def publishBulkOperationEvent(releaseBulkEvent: ReleaseBulkEvent): Unit = {
    eventBus.publish(releaseBulkEvent)
  }

  def abortRelease(releaseId: String, abortComment: String): Release = askAndAwait(releaseId) {
    ExecutionServiceMessages.AbortRelease(abortComment)
  }

  def abortReleases(releaseIds: JUList[String], abortComment: String): JUList[Release] = {
    val releases = askMultipleAndAwait[Release](releaseIds.asScala.toSeq) { _ =>
      ExecutionServiceMessages.AbortRelease(abortComment, isPartOfBulkOperation = true)
    }.asJava
    if (!releases.isEmpty()) {
      publishBulkOperationEvent(ReleaseBulkAbortedEvent(releases))
    }
    releases
  }

  def updateRelease(releaseId: String, release: Release): Release = askAndAwait(releaseId) {
    UpdateRelease(release)
  }

  def deleteRelease(releaseId: String): Unit = askAndAwait(releaseId) {
    DeleteRelease()
  }

  def movePhase(releaseId: String, movementIndexes: MovementIndexes): Phase = askAndAwait(releaseId) {
    MovePhase(movementIndexes)
  }

  def addTeam(releaseId: String, team: Team): Team = askAndAwait(releaseId) {
    AddTeam(team)
  }

  def updateTeam(teamId: String, team: Team): Team = askAndAwait(teamId) {
    UpdateTeam(teamId, team)
  }

  def deleteTeam(teamId: String): Unit = askAndAwait(teamId) {
    DeleteTeam(teamId)
  }

  def createTask(containerId: String, task: Task): Task = createTask(containerId, task, null)

  def createTask(containerId: String, task: Task, position: Integer): Task = askAndAwait(containerId) {
    AddTask(containerId, task, Option(position))
  }

  def moveTask(releaseId: String, movementIndexes: MovementIndexes): Task = askAndAwait(releaseId) {
    MoveTask(movementIndexes)
  }

  def reassignTaskToOwner(taskId: String, owner: String): Task = askAndAwait(taskId) {
    ReassignTaskToOwner(taskId, owner)
  }

  def changeTaskType(taskId: String, taskType: Type): Task = askAndAwait(taskId) {
    ChangeTaskType(taskId, taskType)
  }

  def completeTask(taskId: String, comment: String): Task = askAndAwait(taskId) {
    CompleteTask(taskId, comment)
  }

  def completeTasks(taskIds: JUList[String], comment: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    CompleteTasks(taskIds.toList, comment)
  }.flatMap(_.asScala).asJava

  def failTaskManually(taskId: String, comment: String): Task = askAndAwait(taskId) {
    FailTaskManually(taskId, comment)
  }

  def failTasksManually(taskIds: JUList[String], comment: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    FailTasksManually(taskIds.toList, comment)
  }.flatMap(_.asScala).asJava

  def skipTask(taskId: String, comment: String, user: User = User.AUTHENTICATED_USER): Task = askAndAwait(taskId) {
    SkipTask(taskId, comment, user)
  }

  def skipTasks(taskIds: JUList[String], comment: String, user: User = User.AUTHENTICATED_USER): JUList[String] =
    askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
      SkipTasks(taskIds.toList, comment, user)
    }.flatMap(_.asScala).asJava

  def updateTask(taskId: String, task: Task): Task =
    updateTask(taskId, task, Set[TaskUpdateDirective]().asJava, overrideLock = false)

  def updateTask(taskId: String, task: Task, updateDirectives: java.util.Set[TaskUpdateDirective]): Task =
    updateTask(taskId, task, updateDirectives, overrideLock = false)

  def updateTask(taskId: String, task: Task, updateDirectives: java.util.Set[TaskUpdateDirective], overrideLock: Boolean): Task = askAndAwait(taskId) {
    UpdateTask(taskId, task, updateDirectives, overrideLock)
  }

  def updateTaskStatusLine(taskId: String, statusLine: String): Unit = tell(taskId) {
    UpdateTaskStatusLine(taskId, statusLine)
  }

  def abortTask(taskId: String, comment: String): Task = askAndAwait(taskId) {
    ExecutionServiceMessages.AbortTask(taskId, comment)
  }

  def deleteTask(taskId: String): Unit = askAndAwait(taskId) {
    DeleteTask(taskId)
  }

  def deleteTasks(taskIds: JUList[String]): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    DeleteTasks(taskIds.toList)
  }.flatMap(_.asScala).asJava

  def reassignTaskToTeam(taskId: String, team: String): Unit = askAndAwait(taskId) {
    ReassignTaskToTeam(taskId, team)
  }

  def reassignTasks(taskIds: JUList[String], team: String, owner: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    ReassignTasks(taskIds.toList, team, owner)
  }.flatMap(_.asScala).asJava

  def reopenTask(taskId: String, comment: String): Task = askAndAwait(taskId) {
    ExecutionServiceMessages.ReopenTask(taskId, comment)
  }

  def retryTask(taskId: String, comment: String): Task = askAndAwait(taskId) {
    ExecutionServiceMessages.RetryTask(taskId, comment)
  }

  def retryTasks(taskIds: JUList[String], comment: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    ExecutionServiceMessages.RetryTasks(taskIds.toList, comment)
  }.flatMap(_.asScala).asJava

  def abortTasks(taskIds: JUList[String], comment: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    ExecutionServiceMessages.AbortTasks(taskIds.toList, comment)
  }.flatMap(_.asScala).asJava

  def reopenTasks(taskIds: JUList[String], comment: String): JUList[String] = askMultipleAndAwait[JUList[String]](taskIds.asScala.toSeq) { taskIds =>
    ExecutionServiceMessages.ReopenTasks(taskIds.toList, comment)
  }.flatMap(_.asScala).asJava

  def startTask(taskId: String, comment: String): Task = askAndAwait(taskId) {
    ExecutionServiceMessages.StartTask(taskId, comment)
  }

  def startTaskAsync(taskId: String, comment: String, user: User = User.AUTHENTICATED_USER): Unit = tell(taskId) {
    ExecutionServiceMessages.StartTaskAsync(taskId, comment, user)
  }

  def startTaskWithInput(taskId: String, inputs: util.List[Variable]): Task = askAndAwait(taskId) {
    ExecutionServiceMessages.StartTaskWithInput(taskId, inputs.asScala.toList)
  }

  def lockTask(taskId: String): Task = askAndAwait(taskId) {
    LockTask(taskId)
  }

  def unlockTask(taskId: String): Task = askAndAwait(taskId) {
    UnlockTask(taskId)
  }

  def addPhase(releaseId: String): Phase = askAndAwait(releaseId) {
    AddPhase(None, None)
  }

  def addPhase(releaseId: String, phase: Phase, position: Integer): Phase = askAndAwait(releaseId) {
    AddPhase(Some(phase), Some(position))
  }

  def copyPhase(releaseId: String, phaseIdToCopy: String, targetPosition: Int): Phase = askAndAwait(releaseId) {
    CopyPhase(phaseIdToCopy, targetPosition)
  }

  def updatePhase(id: String, phase: Phase): Phase = askAndAwait(id) {
    UpdatePhase(id, phase)
  }

  def deletePhase(phaseId: String): Unit = askAndAwait(phaseId) {
    DeletePhase(phaseId)
  }

  def copyTask(taskId: String, targetContainerId: String, targetPosition: Int): Task = askAndAwait(taskId) {
    CopyTask(taskId, targetContainerId, targetPosition)
  }

  def duplicateTask(id: String, taskId: String): Task = askAndAwait(id) {
    DuplicateTask(taskId)
  }

  def duplicatePhase(id: String, phaseId: String): Phase = askAndAwait(id) {
    DuplicatePhase(phaseId)
  }

  def updateVariables(releaseId: String, vars: util.List[Variable]): Release = askAndAwait(releaseId) {
    UpdateVariables(vars.asScala.toList)
  }

  def updateTeams(releaseId: String, teams: util.List[Team]): util.List[Team] = askAndAwait(releaseId) {
    UpdateTeams(teams.asScala.toList)
  }

  def restartPhase(releaseId: String,
                   phaseId: String,
                   taskId: String,
                   phaseVersion: PhaseVersion): Release = restartPhase(releaseId, phaseId, taskId, phaseVersion, resumeRelease = false)

  def restartPhase(releaseId: String,
                   phaseId: String,
                   taskId: String,
                   phaseVersion: PhaseVersion,
                   resumeRelease: Boolean): Release = askAndAwait(releaseId) {
    RestartPhase(phaseId, taskId, phaseVersion, resumeRelease)
  }

  def resume(releaseId: String): Release = askAndAwait(releaseId) {
    ExecutionServiceMessages.ResumeRelease()
  }

  def deleteAttachment(id: String, attachmentId: String): Unit = askAndAwait(id) {
    DeleteAttachment(attachmentId)
  }

  def deleteAttachmentFromTask(id: String, taskId: String, attachmentId: String): Unit = askAndAwait(id) {
    DeleteAttachmentFromTask(taskId, attachmentId)
  }

  def createAttachmentOnRelease(releaseId: String, attachment: Attachment): Attachment = askAndAwait(releaseId) {
    CreateAttachmentOnRelease(attachment)
  }

  def createAttachmentOnTask(taskId: String, attachment: Attachment): Attachment = askAndAwait(taskId) {
    CreateAttachmentOnTask(taskId, attachment)
  }

  def createReleaseFromTrigger(templateId: String
                               , folderId: String
                               , releaseTitle: String
                               , releaseVariables: java.util.List[Variable]
                               , releaseTags: java.util.Set[String]
                               , autoStart: Boolean
                               , scheduledStartDate: Option[Date] = None
                               , triggerId: String): CreateReleaseResult = askAndAwait(templateId) {
    CreateReleaseFromTrigger(folderId, releaseTitle, releaseVariables.asScala.toSeq, releaseTags.asScala.toList, autoStart, scheduledStartDate, triggerId)
  }

  def createDependency(parentId: String, targetIdOrVariable: String): Dependency = askAndAwait(parentId) {
    AddDependency(parentId, targetIdOrVariable)
  }

  def updateDependency(id: String, dependencyId: String, targetIdOrVariable: String): Dependency = askAndAwait(id) {
    UpdateDependency(dependencyId, targetIdOrVariable)
  }

  def deleteDependency(id: String, dependencyId: String): Unit = askAndAwait(id) {
    DeleteDependency(dependencyId)
  }

  def archiveDependencies(id: String, dependencyIds: Seq[String]): Unit = askAndAwait(id) {
    ArchiveDependencies(dependencyIds)
  }

  def updateDates(planItemId: String, scheduledStartDate: Date, dueDate: Date, plannedDuration: Integer): Unit = askAndAwait(planItemId) {
    UpdatePlanItemDates(planItemId, scheduledStartDate, dueDate, plannedDuration)
  }

  def deleteTemplate(templateId: String): Unit = askAndAwait(templateId) {
    DeleteTemplate()
  }

  def updateTemplate(templateId: String, template: Release): Release = askAndAwait(templateId) {
    UpdateTemplate(template)
  }

  def createGateCondition(gateId: String): GateCondition = askAndAwait(gateId) {
    CreateGateCondition(gateId)
  }

  def updateGateCondition(conditionId: String, gateCondition: GateCondition): GateCondition = askAndAwait(conditionId) {
    UpdateGateCondition(conditionId, gateCondition)
  }

  def deleteGateCondition(conditionId: String): Unit = askAndAwait(conditionId) {
    DeleteGateCondition(conditionId)
  }

  def createLink(containerId: String, sourceId: String, targetId: String): Link = askAndAwait(containerId) {
    CreateLink(containerId, sourceId, targetId)
  }

  def deleteLink(linkId: String): Unit = askAndAwait(linkId) {
    DeleteLink(linkId)
  }

  def markTaskAsDone(targetStatus: TaskStatus, taskId: String, comment: String, user: User): Unit = {
    markTaskAsDone(targetStatus, taskId, comment, None, user)
  }

  def markTaskAsDone(targetStatus: TaskStatus, taskId: String, comment: String, artifactId: Option[String], user: User): Unit = askAndAwait(taskId) {
    MarkTaskAsDone(taskId, targetStatus, comment, user, artifactId)
  }

  def markTaskAsDoneAsync(targetStatus: TaskStatus, taskId: String, comment: String, user: User): Unit = tell(taskId) {
    MarkTaskAsDone(taskId, targetStatus, comment, user, None)
  }

  def finishContainerTask(result: ContainerTaskResult): Unit = askAndAwait(result.taskId) {
    FinishContainerTask(result)
  }

  def finishCustomScriptTask(taskId: String, executionId: String, executionLog: String, attachmentId: Option[String], results: Option[CustomScriptTaskResults]): Unit = askAndAwaitWithRetry(taskId) {
    FinishCustomScriptTask(taskId, executionLog, executionId, attachmentId, results)
  }

  def finishScriptTask(taskId: String, executionId: String, executionLog: String, attachmentId: Option[String], results: Option[ScriptTaskResults]): Unit = askAndAwaitWithRetry(taskId) {
    FinishScriptTask(taskId, executionLog, executionId, attachmentId, results)
  }

  def markTaskAsDoneWithScriptResults(targetStatus: TaskStatus, taskId: String, comment: String, artifactId: Option[String], baseScriptTaskResults: Option[BaseScriptTaskResults], user: User, executionId: String): Unit = askAndAwaitWithRetry(taskId) {
    MarkTaskAsDoneWithScriptResults(taskId, targetStatus, comment, artifactId, baseScriptTaskResults, user, executionId)
  }

  def failTask(taskId: String, comment: String, user: User, taskResults: Option[BaseScriptTaskResults] = None): Unit = askAndAwait(taskId) {
    FailTask(taskId, comment, taskResults, user)
  }

  def failTasks(taskIds: Seq[String], comment: String): Seq[String] = askMultipleAndAwait[Seq[String]](taskIds) { taskIds =>
    FailTasks(taskIds, comment)
  }.flatten

  def failTaskWithRetry(taskId: String, comment: String, user: User, taskResults: Option[BaseScriptTaskResults] = None, executionId: String): Unit = askAndAwaitWithRetry(taskId) {
    FailTask(taskId, comment, taskResults, user, Some(executionId))
  }

  def failScriptTask(taskId: String, scriptFailMessage: String, scriptExecutionId: String, logArtifactId: Option[String], baseScriptTaskResults: Option[BaseScriptTaskResults], user: User = User.LOG_OUTPUT): Unit = askAndAwaitWithRetry(taskId) {
    FailScriptTask(taskId, scriptFailMessage, scriptExecutionId, logArtifactId, baseScriptTaskResults, user)
  }

  def failTaskAsync(taskId: String, comment: String, user: User, baseScriptTaskResults: Option[BaseScriptTaskResults] = None): Unit = tell(taskId) {
    FailTask(taskId, comment, baseScriptTaskResults, user)
  }

  def taskPreconditionValidated(taskId: String, executionId: String): Unit = askAndAwaitWithRetry(taskId) {
    TaskPreconditionValidated(taskId, executionId)
  }

  def createVariable(variable: Variable, releaseId: String): Variable = askAndAwait(releaseId) {
    CreateVariable(variable)
  }

  def updateVariable(variableId: String, variable: Variable): Variable = askAndAwait(variableId) {
    UpdateVariable(variableId, variable)
  }

  def deleteVariable(variableId: String): Unit = askAndAwait(variableId) {
    DeleteVariable(variableId)
  }

  def replaceVariable(variable: Variable, replacement: VariableOrValue): Unit = askAndAwait(variable.getId) {
    ReplaceVariable(variable, replacement)
  }

  def renameVariable(variableId: String, variable: Variable): Variable = askAndAwait(variableId) {
    RenameVariable(variableId, variable)
  }

  def executeCommand[T <: ExtensionCommand, S](ciId: String, command: T): S = askAndAwait(ciId) {
    command
  }

  def executeCommandAsync[T <: ExtensionCommand](ciId: String, command: T): Unit = tell(ciId) {
    command
  }

  def notifyOverdue(releaseId: String): Unit = tell(releaseId) {
    ReleaseOverdue()
  }

  @varargs
  def notifyOverdueTasks(releaseId: String, taskIds: String*): Unit = tell(releaseId) {
    TasksOverdue(taskIds)
  }

  @varargs
  def notifyDueSoonTasks(releaseId: String, taskIds: String*): Unit = tell(releaseId) {
    TasksDueSoon(taskIds)
  }

  def postponeTaskUntilEnvironmentsAreReserved(releaseId: String, taskId: String, postponeUntil: Date, comment: String, executionId: String): Unit = askAndAwaitWithRetry(releaseId) {
    PostponeTaskUntilEnvironmentsAreReserved(taskId, postponeUntil, comment, executionId)
  }

  def handleCreateReleaseTaskExecutionResult(result: CreateReleaseTaskExecutionResult): Unit = tell(result.taskId) {
    ProcessCreateReleaseTaskExecutionResult(result)
  }

  object Timeout {}

  private def askFutureRetryLoop(releaseId: String, msg: AnyRef, completableFuture: CompletableFuture[Any], askTimeout: Timeout, waitForAnyResponse: Boolean = true, tries: Int = 0, futures: Seq[Future[Any]] = Seq()): Unit = {
    implicit val remainingTimeout: Timeout = config.timeouts.messageRetryTimeout.mul(config.timeouts.messageRetryAttempts - tries)
    logger.debug(s"askFutureRetryLoop[${tries}] ${msg}")
    val future = releaseActorRef ? ReleaseAction(releaseId, msg)
    val timeoutFuture = new CompletableFuture[Any]()
    timeoutExecutor.schedule(() => timeoutFuture.complete(Timeout), askTimeout.duration.toMillis, TimeUnit.MILLISECONDS)
    val futuresToWaitFor = futures ++ Seq(future, timeoutFuture.asScala)
    implicit val executionContext = scala.concurrent.ExecutionContext.Implicits.global
    val result = Future.firstCompletedOf(futuresToWaitFor)
    result.onComplete {
      case Failure(e: TimeoutException) if tries < config.timeouts.messageRetryAttempts =>
        logger.debug(s"askFutureRetryLoop[$tries] got TimeoutException ${msg}")
        // looks like thread starvation, we have got TimeoutException but we still have retries.
        // there is no use to recursively pass futures to us, because at least one of them already
        // is failed with TimeoutException
        askFutureRetryLoop(releaseId, msg, completableFuture, askTimeout, waitForAnyResponse, tries + 1)
      case Success(Timeout) =>
        logger.debug(s"askFutureRetryLoop[$tries] got retry Timeout ${msg}")
        askFutureRetryLoop(releaseId, msg, completableFuture, askTimeout, waitForAnyResponse, tries + 1, futures ++ Seq(future))
      case Failure(e: TimeoutException) =>
        logger.error(s"askFutureRetryLoop timeout, ${msg}")
        completableFuture.completeExceptionally(e)
      case Failure(e: Throwable) =>
        logger.error(s"Got exception in askFutureRetryLoop ${msg}", e)
        if (waitForAnyResponse) {
          completableFuture.completeExceptionally(e)
        } else {
          Future.foldLeft(futuresToWaitFor)(true)((_, _) => true).onComplete(_ => completableFuture.completeExceptionally(e))
        }
      case Success(response) =>
        logger.debug(s"askFutureRetryLoop succeeded ${msg}")
        if (waitForAnyResponse) {
          completableFuture.complete(response)
        } else {
          Future.foldLeft(futuresToWaitFor)(true)((_, _) => true).onComplete(_ => completableFuture.complete(response))
        }
    }(scala.concurrent.ExecutionContext.Implicits.global)
  }

  @Subscribe
  def onReleaseStatusChange(event: ReleaseExecutionEvent): Unit =
    tell(event.release.getId) {
      StatusUpdated(event.release.getStatus)
    }

  override def destroy(): Unit = {
    logger.info("Shutting down Release actor service")
    ExecutorServiceUtils.shutdown("timeout executor", timeoutExecutor)
  }

  def sendBackpressure(taskId: String): Future[Any] = {
    implicit val askTimeout: Timeout = config.timeouts.releaseActionResponse
    releaseActorRef ? ReleaseAction(releaseIdFrom(taskId), BackpressureWait(taskId))
  }

}

// scalastyle:on number.of.methods
