package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.plugin.api.reflect.{Descriptor, DescriptorRegistry, Type}
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters
import com.xebialabs.xlrelease.domain.events.{FacetConfiguredFacetsCreatedEvent, FacetCreatedEvent, FacetDeletedEvent, FacetUpdatedEvent}
import com.xebialabs.xlrelease.domain.facet.FacetScope.TASK
import com.xebialabs.xlrelease.domain.facet.{Facet, TaskReportingRecord}
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.{isTaskId, releaseIdFrom}
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.utils.TypeHelper.getAllSubtypesOf
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{Collections, List => JList}
import scala.jdk.CollectionConverters._

@Service
class FacetService @Autowired()(facetRepositoryDispatcher: FacetRepositoryDispatcher,
                                taskRepository: TaskRepository,
                                eventBus: XLReleaseEventBus,
                                archivingService: ArchivingService,
                                releaseRepository: ReleaseRepository,
                                ciIdService: CiIdService,
                                variableRepository: ReleaseVariableRepository,
                                releaseActorService: ReleaseActorService,
                                taskAccessService: TaskAccessService,
                                @Autowired(required = false) facetConfigurationService: FacetConfigurationService
                               ) extends Logging {

  // TODO: add logging statements to service

  @Timed
  def get(facetId: String): Facet = facetRepositoryDispatcher.get(facetId)

  @Timed
  def addTaskReportingRecord(record: TaskReportingRecord, applyTaskAttributes: Boolean): java.util.List[TaskReportingRecord] = {
    if (applyTaskAttributes) {
      this.createApplyingConfiguration(record, createUnconfigured = true)
    } else {
      Collections.singletonList(this.create(record).asInstanceOf[TaskReportingRecord])
    }
  }

  @Timed
  def create(facet: Facet): Facet = {
    validate(facet)
    val task: Task = taskRepository.findById(facet.getTargetId, ResolveOptions.WITH_DECORATORS)
    taskAccessService.checkIfAuthenticatedUserCanUseTask(task)
    facet.setId(Ids.getName(ciIdService.getUniqueId(Type.valueOf(classOf[Facet]), "")))
    val createdFacet = facetRepositoryDispatcher.create(facet)

    createNewVariables(task.getRelease, createdFacet)
    eventBus.publish(FacetCreatedEvent(createdFacet))
    createdFacet
  }

  @Timed
  def createApplyingConfiguration(facet: TaskReportingRecord, createUnconfigured: Boolean): java.util.List[TaskReportingRecord] = {
    val (hasConfiguration, facets) = if (facetConfigurationService != null) {
      facetConfigurationService.configureFacet(facet)
    } else {
      false -> Seq(facet)
    }
    if (createUnconfigured || hasConfiguration) {
      facets.foreach(create)
      eventBus.publish(FacetConfiguredFacetsCreatedEvent(facet, facets.asJava, hasConfiguration))
      facets.asJava
    } else {
      java.util.Collections.emptyList()
    }
  }

  @Timed
  def delete(facetId: String): Unit = {
    val facet = facetRepositoryDispatcher.get(facetId)
    val taskBasicData = taskRepository.findTaskBasicData(facet.getTargetId)
    taskAccessService.checkIfAuthenticatedUserCanUseTaskType(taskBasicData.taskType)
    facetRepositoryDispatcher.delete(facetId)
    eventBus.publish(FacetDeletedEvent(facet))
  }

  @Timed
  def update(facet: Facet): Facet = {
    validate(facet)

    val task: Task = taskRepository.findById(facet.getTargetId, ResolveOptions.WITH_DECORATORS)
    taskAccessService.checkIfAuthenticatedUserCanUseTask(task)
    val originalFacet = facetRepositoryDispatcher.get(facet.getId)
    val updatedFacet = facetRepositoryDispatcher.update(facet)

    createNewVariables(task.getRelease, updatedFacet)
    eventBus.publish(FacetUpdatedEvent(originalFacet, updatedFacet))
    updatedFacet
  }

  @Timed
  def isImmutable(facet: Facet): Boolean = facet.getType.instanceOf(Type.valueOf(classOf[TaskReportingRecord]))

  @Timed
  def search(filters: FacetFilters): JList[Facet] = {
    filters.setTypes(Option(filters.getTypes).map(_.asScala.flatMap(getAllSubtypesOf)).getOrElse(Seq.empty).asJava)
    val containerId = if (filters.getParentId != null) filters.getParentId else filters.getTargetId

    if (archivingService.exists(releaseIdFrom(containerId))) {
      logger.debug(s"Search in ARCHIVE for ${filters.getParentId}/${filters.getTargetId}/${filters.getTypes}")
      facetRepositoryDispatcher.archiveRepository.search(filters).asJava
    } else {
      logger.debug(s"Search in LIVE for ${filters.getParentId}/${filters.getTargetId}/${filters.getTypes}")
      facetRepositoryDispatcher.liveRepository.search(filters).asJava
    }
  }

  @Timed
  def exists(facetId: String): Boolean = facetRepositoryDispatcher.exists(facetId)

  private def createNewVariables(release: Release, facet: Facet): Unit = {
    val newVariables = scanAndBuildNewVariables(release, facet, ciIdService).asScala
    if (newVariables.nonEmpty) {
      newVariables.foreach(v => releaseActorService.createVariable(v, release.getId))
    }
  }

  @Timed
  def getFacetTypes(baseType: String): JList[Descriptor] = DescriptorRegistry.getSubtypes(Type.valueOf(baseType))
    .asScala
    .map(_.getDescriptor)
    .filter(desc => !desc.isVirtual)
    .filterNot(desc => desc.getType.instanceOf(Type.valueOf(classOf[TaskReportingRecord])))
    .toList
    .sortBy(_.getLabel).asJava

  private def validate(facet: Facet): Unit = {
    facet.getScope match {
      case TASK =>
        // TODO: if necessary, validate if the target entity is updateable, if it is not, then the faces also can't be created/updated
        checkArgument(isTaskId(facet.getTargetId), s"Target ID is not a valid task ID.")
        checkArgument(taskRepository.exists(facet.getTargetId),
          s"Task with the ID '${facet.getTargetId}' does not exist or has been archived.")
      case _ => throw new IllegalArgumentException(s"Target type ${facet.getScope} is not supported.")
    }
  }

}
