package com.xebialabs.xlrelease.api.v1.impl;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import jakarta.ws.rs.NotFoundException;
import jakarta.ws.rs.core.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.xlrelease.api.v1.UserApi;
import com.xebialabs.xlrelease.api.v1.forms.UserAccount;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.service.Users;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.views.ChangePasswordView;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.api.internal.ProfileResource.InvalidPreviousPasswordException;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;
import static java.lang.String.format;


@Controller
public class UserApiImpl implements UserApi {

    private static final Logger logger = LoggerFactory.getLogger(UserApiImpl.class);

    private final UserProfileService userProfileService;
    private final PermissionChecker permissions;
    private final Users users;
    private final UserService userService;
    private final SessionService sessionService;
    private final UserAccountValidator userAccountValidator;

    @Autowired
    public UserApiImpl(UserProfileService userProfileService,
                   PermissionChecker permissions,
                   Users users,
                   UserService userService,
                   SessionService sessionService,
                   UserAccountValidator userAccountValidator) {
        this.userProfileService = userProfileService;
        this.permissions = permissions;
        this.users = users;
        this.userService = userService;
        this.sessionService = sessionService;
        this.userAccountValidator = userAccountValidator;
    }

    // REST API

    @Timed
    @Override
    public Response createUser(String username, UserAccount userAccount) {
        userAccount.setUsername(username);
        UserAccount createdAccount = createUser(userAccount);
        return Response.ok(createdAccount).build();
    }

    @Timed
    @Override
    public Response updateUser(String username, UserAccount userAccount) {
        userAccount.setUsername(username);
        userAccountValidator.checkEmail(userAccount);
        UserAccount updatedAccount = updateUser(userAccount);
        return Response.ok(updatedAccount).build();
    }

    @Timed
    @Override
    public Response updateUsersRest(Collection<UserAccount> userAccounts) {
        updateUsers(userAccounts);
        return Response.noContent().build();
    }

    @Timed
    @Override
    public Response updatePassword(String username, ChangePasswordView changePasswordView) {
        updatePassword(username, changePasswordView.getCurrentPassword(), changePasswordView.getNewPassword());
        return Response.noContent().build();
    }

    @Timed
    @Override
    public Response deleteUserRest(String username) {
        deleteUser(username);
        return Response.noContent().build();
    }


    // REST+Jython API

    @Timed
    @Override
    public List<UserAccount> findUsers(String email, String fullName, Boolean loginAllowed, Boolean external, Date lastActiveAfter, Date lastActiveBefore, Long page, Long resultsPerPage) {
        permissions.check(EDIT_SECURITY);
        final List<UserProfile> userProfiles = userProfileService.search(email, fullName, loginAllowed, lastActiveAfter, lastActiveBefore, page, resultsPerPage);
        List<String> internalNames = users.getRepositoryUsernames();
        return userProfiles.stream()
                .map(profile -> new UserAccount(profile.getCanonicalId(), profile, isInternalUser(internalNames, profile.getCanonicalId())))
                .filter(account -> external == null || account.isExternal() == external)
                .collect(Collectors.toList());
    }

    @Timed
    @Override
    public UserAccount getUser(String username) {
        if (isUsernameLoggedIn(username) || permissions.hasGlobalPermission(EDIT_SECURITY)) {
            final Optional<UserProfile> byUsername = Optional.ofNullable(userProfileService.findByUsername(username));
            final Optional<UserAccount> userAccount = byUsername
                    .flatMap(u -> Optional.of(new UserAccount(username, u, users.userExistsInRepository(username))));
            return userAccount.orElseThrow(() -> new NotFoundException(format("User [%s] not found", username)));
        } else {
            throw PermissionDeniedException.forPermission(EDIT_SECURITY, (String) null);
        }
    }


    // Jython API

    @Timed
    @Override
    public UserAccount createUser(UserAccount userAccount) {
        permissions.check(EDIT_SECURITY);
        userAccountValidator.check(userAccount);

        logger.info(format("Creating user [%s]", userAccount.getUsername()));

        UserProfile userProfile = userAccount.toUserProfile();
        userProfileService.validate(userProfile);
        userService.create(userAccount.getUsername(), userAccount.getPassword());
        userProfileService.save(userProfile);

        return getUser(userAccount.getUsername());
    }

    @Timed
    @Override
    public UserAccount updateUser(UserAccount userAccount) {
        String username = userAccount.getUsername();
        if (isUsernameLoggedIn(username) || permissions.hasGlobalPermission(EDIT_SECURITY)) {
            logger.info(format("Updating user [%s]", username));
            UserProfile profile = userAccount.toUserProfile();
            userProfileService.updateProfile(profile);
            return userAccount;
        } else {
            throw PermissionDeniedException.forPermission(EDIT_SECURITY, (String) null);
        }
    }

    @Timed
    @Override
    public void updateUsers(Collection<UserAccount> userAccounts) {
        permissions.check(EDIT_SECURITY);
        userAccounts.forEach(user -> {
            userAccountValidator.checkEmail(user);
            checkArgument(isNotBlank(user.getUsername()), "Username is required");
        });

        logger.info(format("Updating %d users", userAccounts.size()));

        UserProfile[] userProfilesToUpdate = userAccounts.stream()
                .map(UserAccount::toUserProfile)
                .toArray(UserProfile[]::new);

        userProfileService.updateProfile(userProfilesToUpdate);
    }

    @Timed
    @Override
    public void updatePassword(String username, String currentPassword, String newPassword) {
        boolean canEditSecurity = permissions.hasGlobalPermission(EDIT_SECURITY);
        if (!canEditSecurity && !isUsernameLoggedIn(username)) {
            throw PermissionDeniedException.forPermission(EDIT_SECURITY, (String) null);
        }

        if (users.userExistsInRepository(username)) {
            userAccountValidator.checkPassword(newPassword);
            try {
                checkArgument(isNotBlank(newPassword), "Password cannot be empty.");
                logger.info(format("Updating password of user [%s]", username));
                if (canEditSecurity && !isUsernameLoggedIn(username)) {
                    userService.modifyPassword(username, newPassword);
                } else {
                    checkArgument(isNotBlank(currentPassword), "currentPassword must be provided when changing the password.");
                    userService.modifyPassword(username, newPassword, currentPassword);
                }
                sessionService.disconnect(username);
            } catch (IllegalArgumentException e) {
                throw new InvalidPreviousPasswordException(e);
            }
        } else {
            throw new Checks.IncorrectArgumentException("Cannot update a password of an external user");
        }
    }

    @Timed
    @Override
    public void deleteUser(String username) {
        permissions.check(EDIT_SECURITY);
        checkArgument(!isAdmin(username), "Admin user cannot be deleted.");
        checkArgument(!isUsernameLoggedIn(username), "Your own account cannot be deleted.");

        // Check if user exists and throw error
        getUser(username);
        logger.info(format("Deleting user [%s]", username));
        if (users.userExistsInRepository(username)) {
            userService.delete(username);
        }
        userProfileService.deleteByUsername(username);
        sessionService.disconnect(username);
    }

    private boolean isUsernameLoggedIn(String username) {
        return username.equalsIgnoreCase(User.AUTHENTICATED_USER.getName());
    }

    private boolean isInternalUser(List<String> internalNames, String name) {
        for (String internalName : internalNames) {
            if (name.equalsIgnoreCase(internalName)) {
                return true;
            }
        }
        return false;
    }
}
