package com.xebialabs.xlrelease.service;

import com.codahale.metrics.annotation.Timed;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.io.StreamWrappingOverthereFile;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.configuration.CustomLogoContentType;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.events.TemplateLogoUpdatedEvent;
import com.xebialabs.xlrelease.domain.utils.IdUtils;
import com.xebialabs.xlrelease.events.EventBus;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.utils.FileContentValidation;
import jakarta.activation.FileTypeMap;
import jakarta.servlet.http.Part;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import static java.lang.String.format;
import static scala.jdk.javaapi.OptionConverters.toJava;

@Service
public class UploadService {

    private static final Logger logger = LoggerFactory.getLogger(UploadService.class);
    public static final String NEWLINES = "[\n\r]";
    private final AttachmentService attachmentService;
    private final ReleaseRepository releaseRepository;
    private final AttachmentRepository attachmentRepository;
    private final ReleaseActorService releaseActorService;
    private final TemplateMetadataRepository templateMetadataRepository;
    private final EventBus eventBus;

    @Autowired
    public UploadService(AttachmentService attachmentService,
                         ReleaseRepository releaseRepository,
                         AttachmentRepository attachmentRepository,
                         ReleaseActorService releaseActorService,
                         TemplateMetadataRepository templateMetadataRepository,
                         EventBus eventBus) {
        this.attachmentService = attachmentService;
        this.releaseRepository = releaseRepository;
        this.attachmentRepository = attachmentRepository;
        this.releaseActorService = releaseActorService;
        this.templateMetadataRepository = templateMetadataRepository;
        this.eventBus = eventBus;
    }

    @Timed
    public List<Attachment> addAttachment(String ciId, Iterator<Part> attachments) throws IOException {
        List<Attachment> attachmentList = new ArrayList<>();

        while (attachments.hasNext()) {
            Part item = attachments.next();
            String filename = item.getSubmittedFileName();
            String contentType = item.getContentType();
            try (InputStream stream = item.getInputStream();
                 InputStream validatedInputStream = FileContentValidation.apply(filename, stream).get()
            ) {
                logger.debug("Attaching {} to {}", filename.replaceAll(NEWLINES, "_"), ciId.replaceAll(NEWLINES, "_"));
                attachmentList.add(addAttachment(ciId, filename, contentType, validatedInputStream));
            }
        }

        return attachmentList;
    }

    @Timed
    public TemplateLogo addLogo(String ciId, Iterator<Part> logo) throws IOException {
        if (logo.hasNext()) {
            Part item = logo.next();
            String filename = item.getSubmittedFileName();
            String contentType = item.getContentType();
            if (!CustomLogoContentType.getAllTypes().contains(contentType)) {
                throw new IllegalArgumentException(format("Content type %s not allowed. Allowed content types are: %s", contentType, CustomLogoContentType.getBrowserTypes().toString()));
            }
            try (InputStream stream = item.getInputStream();
                 InputStream validatedInputStream = FileContentValidation.apply(filename, stream).get()
            ) {
                return addLogo(ciId, filename, contentType, validatedInputStream);
            }
        }

        return null;
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, byte[] fileByteArray) {
        return addAttachment(ciId, fileName, FileTypeMap.getDefaultFileTypeMap().getContentType(fileName), new ByteArrayInputStream(fileByteArray));
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, String contentType, InputStream inputStream) {
        logger.debug("Attaching {} to {}", fileName.replaceAll(NEWLINES, "_"), ciId.replaceAll(NEWLINES, "_"));
        Release release = releaseRepository.findById(Ids.releaseIdFrom(ciId));
        Task task = null;
        if (!Ids.isReleaseId(ciId)) {
            task = release.getTask(ciId);
        }

        Attachment attachment = attachmentService.attachToRelease(release, fileName, contentType, inputStream);
        if (task != null) {
            task.getAttachments().add(attachment);
        }
        attachmentRepository.insertAttachment(release.getId(), attachment);
        if (null != task) {
            return releaseActorService.createAttachmentOnTask(task.getId(), attachment);
        } else {
            return releaseActorService.createAttachmentOnRelease(release.getId(), attachment);
        }
    }

    @Timed
    public TemplateLogo addLogo(String releaseId, String fileName, String contentType, InputStream inputStream) {
        logger.debug("Attaching logo to {}", releaseId);
        Optional<Integer> releaseUid = toJava(releaseRepository.getUid(releaseId));

        if (releaseUid.isPresent()) {
            TemplateLogo logo = createLogo(releaseId, fileName, contentType, inputStream);
            TemplateLogo createdLogo =  templateMetadataRepository.createOrUpdateLogo(releaseUid.get(), logo);
            eventBus.publish(new TemplateLogoUpdatedEvent(releaseId, createdLogo));
            return createdLogo;
        } else {
            throw new NotFoundException("Repository entity [%s] not found", releaseId);
        }
    }

    private TemplateLogo createLogo(String releaseId, String filename, String contentType, InputStream bytes) {
        AttachmentSizeLimiter limiter = new AttachmentSizeLimiter(TemplateLogo.getDefaultMaxLogoSize(), bytes);
        StreamWrappingOverthereFile file = new StreamWrappingOverthereFile(filename, limiter);
        String logoId = IdUtils.getUniqueId(Type.valueOf(TemplateLogo.class), releaseId);

        TemplateLogo templateLogo = new TemplateLogo(file, contentType);
        templateLogo.setId(logoId);
        templateLogo.getExportFilename();

        String releaseTitle = releaseRepository.getTitle(releaseId);
        templateLogo.setProperty(BaseArtifact.PARENT_TITLE_PROPERTY_NAME, releaseTitle);
        templateLogo.getPortableFilename();

        return templateLogo;
    }
}
