package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.scheduler.logs.TaskLogCreated
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

object TaskSSEEventListener {
  private val LOG_EVENT = "TASK_LOG_EVENT"
  private val COMMENT_EVENT = "TASK_COMMENT_EVENT"
  private val ATTACHMENT_EVENT = "TASK_ATTACHMENT_EVENT"
  private val TASK_STATUS_LINE = "TASK_STATUS_LINE"
  private val TASK_EXECUTION_EVENT = "TASK_EXECUTION_EVENT"
}

@Service
@EventListener
class TaskSSEEventListener @Autowired()(sseService: SseService) extends Logging {

  @AsyncSubscribe
  def onEvent(event: DistributedXLReleaseEvent): Unit = {
    event match {
      case TaskLogCreated(taskId, executionId, _) =>
        sendEventToSink(taskId, TaskSSEEventListener.LOG_EVENT, executionId)
      case e: DistributedCommentEvent =>
        sendEventToSink(e.taskId, TaskSSEEventListener.COMMENT_EVENT, "")
      case e: DistributedAttachmentEvent =>
        sendEventToSink(e.containerId, TaskSSEEventListener.ATTACHMENT_EVENT, "")
      case e: DistributedTaskStatusLineUpdated =>
        sendEventToSink(e.taskId, TaskSSEEventListener.TASK_STATUS_LINE, e.statusLine)
      case e: DistributedTaskStatusEvent =>
        sendEventToSink(Ids.releaseIdFrom(e.containerId), TaskSSEEventListener.TASK_EXECUTION_EVENT, e.taskStatus.value())
      case _ =>
      // nothing to do yet
    }
  }

  private def sendEventToSink(taskId: String, eventName: String, message: String): Unit = {
    sseService.sendEvent(taskId, eventName, message)
  }

}
