package com.xebialabs.xlrelease.api.v1.impl;


import java.util.*;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.CacheManager;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.repository.ItemAlreadyExistsException;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PlatformPermissions;
import com.xebialabs.xlrelease.api.v1.RolesApi;
import com.xebialabs.xlrelease.api.v1.views.PrincipalView;
import com.xebialabs.xlrelease.api.v1.views.RoleView;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.ReleaseRoleService;
import com.xebialabs.xlrelease.security.ReleaseRoleService.RoleCreateRequest;
import com.xebialabs.xlrelease.security.ReleaseRoleService.RoleUpdateRequest;
import com.xebialabs.xlrelease.views.RolePrincipalsView;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

@Controller
public class RolesApiImpl implements RolesApi {
    private static final Logger logger = LoggerFactory.getLogger(RolesApiImpl.class);

    private final PermissionChecker permissions;
    private final ReleaseRoleService releaseRoleService;
    private final Optional<CacheManager> cacheManager;

    public RolesApiImpl(PermissionChecker permissions,
                        ReleaseRoleService releaseRoleService,
                        @Qualifier("securityCacheManager") Optional<CacheManager> cacheManager) {
        this.permissions = permissions;
        this.releaseRoleService = releaseRoleService;
        this.cacheManager = cacheManager;
    }

    @Timed
    @Override
    public List<RoleView> getRoles(Integer page, Integer resultsPerPage) {
        logger.trace("Entering getRoles(page={}, resultsPerPage={})", page, resultsPerPage);
        invalidateCachesIfNecessary();
        permissions.checkAny(PlatformPermissions.EDIT_SECURITY, PlatformPermissions.VIEW_SECURITY);

        PageRequest pageable = PageRequest.of(page, resultsPerPage);

        List<RolePrincipalsView> rolePrincipalsViews = releaseRoleService.getGlobalRolePrincipalViews(pageable).getContent();
        // TODO check if we can pass role ids into getAllGlobalRolePermissions
        //  (if there are thousands of roles why fetch all of them instead of the ones on the page?)
        Map<String, Set<String>> allRolePermissions = releaseRoleService.getAllGlobalRolePermissions();
        List<RoleView> result = rolePrincipalsViews.stream().map(rolePrincipalsView -> assembleRoleView(rolePrincipalsView, allRolePermissions)).toList();

        logger.trace("Exiting getRoles(page={}, resultsPerPage={}). Found {} roles. Results:\n{}", page, resultsPerPage, result.size(), result);
        return result;
    }

    @Timed
    @Override
    public RoleView getRole(String roleName) {
        logger.trace("Entering getRole(roleName={})", roleName);
        invalidateCachesIfNecessary();
        permissions.checkAny(PlatformPermissions.EDIT_SECURITY, PlatformPermissions.VIEW_SECURITY);

        RolePrincipalsView rolePrincipalsView = releaseRoleService.getGlobalRolePrincipalsView(roleName);
        String roleId = rolePrincipalsView.getRole().getId();
        Set<String> rolePermissions = releaseRoleService.getRolePermissions(roleId);
        RoleView result = assembleRoleView(rolePrincipalsView, Map.of(roleId, rolePermissions));

        logger.trace("Exiting getRole(roleName={}). Result:\n{}", roleName, result);
        return result;
    }

    @Timed
    @Override
    public void create(String roleName, RoleView roleView) {
        logger.trace("Entering create(roleName={}, role={})", roleName, roleView);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        Checks.checkTrue(roleName.equals(roleView.getName()),
                "Role name '%s' given in the path is not equal to the role '%s' defined in the provided object",
                roleName, roleView.getName());
        create(singletonList(roleView));
        logger.trace("Exiting create(roleName={}, role={})", roleName, roleView);
    }

    @Timed
    @Override
    public void create(List<RoleView> roleViews) {
        logger.trace("Entering create(roles={})", roleViews);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        checkPermissions(roleViews);
        List<String> lowerRoleNames = roleViews.stream().map(roleView -> roleView.getName().trim().toLowerCase()).toList();
        Optional<Role> role = releaseRoleService.getGlobalRoles().stream().filter(r -> lowerRoleNames.contains(r.getName().trim().toLowerCase())).findFirst();
        if (role.isPresent()) {
            throw new ItemAlreadyExistsException("Role '%s' already exist. Maybe you wanted to update it?", role.get().getName());
        }

        List<RoleCreateRequest> createRequests = new ArrayList<>();
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName().trim();
            Set<String> rolePrincipals = roleView.getPrincipals().stream().map(PrincipalView::getUsername).collect(toSet());
            Set<String> rolePermissions = roleView.getPermissions();
            RoleCreateRequest r = new RoleCreateRequest(roleName, rolePrincipals, rolePermissions);
            createRequests.add(r);
        });

        releaseRoleService.createGlobalRoleAndRolePermissions(createRequests.toArray(new RoleCreateRequest[0]));
        logger.trace("Exiting create(roles={})", roleViews);
    }

    @Timed
    @Override
    public void update(String roleName, RoleView roleView) {
        logger.trace("Entering update(roleName={}, roles={})", roleName, roleView);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        roleView.setName(roleName);
        update(singletonList(roleView));
        logger.trace("Exiting update(roleName={}, roles={})", roleName, roleView);
    }

    @Timed
    @Override
    public void update(List<RoleView> roleViews) {
        logger.trace("Entering update(roles={})", roleViews);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        checkPermissions(roleViews);

        List<String> roleNamesToUpdate = roleViews.stream().map(roleView -> roleView.getName().trim().toLowerCase()).toList();
        List<Role> roles = releaseRoleService.getGlobalRoles();
        Map<String, Role> rolesByName = roles.stream().collect(toMap(role -> role.getName().trim().toLowerCase(), identity()));

        Optional<String> firstNonExistent = roleNamesToUpdate.stream().filter(r -> !rolesByName.containsKey(r)).findFirst();
        if (firstNonExistent.isPresent()) {
            throw new LogFriendlyNotFoundException("Role '%s' does not exist. Maybe you wanted to create it?", firstNonExistent.get());
        }

        List<RoleUpdateRequest> updateRequests = new ArrayList<>();
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName().trim();
            String roleId = rolesByName.get(roleName.toLowerCase()).getId();
            Set<String> rolePrincipals = roleView.getPrincipals().stream().map(PrincipalView::getUsername).collect(Collectors.toSet());
            RoleUpdateRequest r = new RoleUpdateRequest(roleId, roleName, rolePrincipals, roleView.getPermissions());
            updateRequests.add(r);
        });

        releaseRoleService.updateGlobalRoleAndRolePermissions(updateRequests.toArray(new RoleUpdateRequest[0]));
        logger.trace("Exiting update(roles={})", roleViews);
    }

    @Timed
    @Override
    public void delete(String roleName) {
        logger.trace("Entering delete(roleName={})", roleName);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);

        releaseRoleService.deleteGlobalRole(roleName);

        logger.trace("Exiting delete(roleName={})", roleName);
    }

    @Timed
    @Override
    public void rename(String roleName, String newName) {
        logger.trace("Entering rename(roleName={}, newName={})", roleName, newName);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);

        if (!releaseRoleService.globalRoleExists(roleName)) {
            throw new LogFriendlyNotFoundException("Role '%s' does not exist. Maybe you wanted to create it first?", roleName);
        }
        if (releaseRoleService.globalRoleExists(newName)) {
            throw new ItemAlreadyExistsException("Role '%s' already exist.", newName);
        }

        releaseRoleService.renameGlobalRole(roleName, newName);

        logger.trace("Exiting rename(roleName={}, newName={})", roleName, newName);
    }

    private RoleView assembleRoleView(RolePrincipalsView rolePrincipalsView, Map<String, Set<String>> rolePermissions) {
        final RoleView roleView = new RoleView();
        String roleId = rolePrincipalsView.getRole().getId();
        roleView.setId(roleId);
        roleView.setName(rolePrincipalsView.getRole().getName());
        roleView.setPrincipals(rolePrincipalsView.getPrincipals().stream().map(userView -> {
            PrincipalView principalView = new PrincipalView();
            principalView.setUsername(userView.getUsername());
            principalView.setFullname(userView.getFullName());
            return principalView;
        }).toList());
        Set<String> rolePermissionsOrDefault = rolePermissions.getOrDefault(roleId, emptySet());
        roleView.setPermissions(new HashSet<>(rolePermissionsOrDefault));
        return roleView;
    }

    private void checkPermissions(final List<RoleView> roleViewList) {
        Set<String> unknownPermissions = new TreeSet<>();
        roleViewList.forEach(roleView -> unknownPermissions.addAll(roleView.getPermissions().stream().filter(p -> Permission.find(p) == null).toList()));
        checkArgument(unknownPermissions.isEmpty(), "Unknown permissions found: '%s'", String.join(", ", unknownPermissions));

        Set<Permission> allPermissions = new HashSet<>();
        roleViewList.forEach(roleView -> allPermissions.addAll(roleView.getPermissions().stream().map(Permission::find).filter(Objects::nonNull).toList()));

        Collection<Permission> notApplicableTo = Permissions.isApplicableTo(allPermissions, GLOBAL_SECURITY_ALIAS(), false);
        checkArgument(notApplicableTo.isEmpty(), "Not applicable permissions found: '%s'", notApplicableTo.stream().map(Permission::getPermissionName).collect(Collectors.joining(", ")));
    }

    private void invalidateCachesIfNecessary() {
        try {
            cacheManager.ifPresent(manager -> manager.getCacheNames().forEach(cache -> manager.getCache(cache).invalidate()));
        } catch (Exception e) {
            logger.warn("Unable to clear security cache before operation", e);
        }
    }

}
