package com.xebialabs.xlrelease.api.internal;

import java.util.*;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.core.rest.api.DtoReader;
import com.xebialabs.deployit.core.rest.api.DtoWriter;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.utils.RolePrincipalEventPublisher;
import com.xebialabs.xlrelease.views.RolePrincipalsView;
import com.xebialabs.xlrelease.views.RoleView;
import com.xebialabs.xlrelease.views.UserView;
import com.xebialabs.xlrelease.views.converters.UserViewConverter;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static java.util.stream.Collectors.toList;

/**
 * The login names or LDAP groups that are part of a security role.
 */
@Path("/roles")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class RolePrincipalsResource {

    private static final Logger logger = LoggerFactory.getLogger(RolePrincipalsResource.class);

    private PermissionChecker permissionChecker;
    private RoleService roleService;
    private TeamService teamService;
    private UserViewConverter userViewConverter;
    private XLReleaseEventBus xlReleaseEventBus;
    private final XlrConfig xlrConfig;

    private DtoWriter dtoWriter = new DtoWriter();
    private DtoReader dtoReader = new DtoReader();

    @Autowired
    public RolePrincipalsResource(PermissionChecker permissionChecker,
                                  RoleService roleService,
                                  TeamService teamService,
                                  UserViewConverter userViewConverter,
                                  XLReleaseEventBus xlReleaseEventBus) {
        this.permissionChecker = permissionChecker;
        this.roleService = roleService;
        this.teamService = teamService;
        this.userViewConverter = userViewConverter;
        this.xlReleaseEventBus = xlReleaseEventBus;
        this.xlrConfig = XlrConfig.getInstance();
    }

    @GET
    @Timed
    @Path("names")
    public List<String> readRoleNames() {
        return this.roleService.getRoles().stream().map(Role::getName).collect(toList());
    }

    @GET
    @Timed
    @Path("principals")
    public List<RolePrincipalsView> readRolePrincipals() {
        permissionChecker.check(EDIT_SECURITY);

        List<RolePrincipals> assignments = getRolePrincipalsList();

        Map<String, UserView> cache = new HashMap<>();

        return assignments.stream().map(rolePrincipal -> getRolePrincipalsView(rolePrincipal, cache)).collect(toList());
    }

    private RolePrincipalsView getRolePrincipalsView(final RolePrincipals rolePrincipal, Map<String, UserView> cache) {
        RoleView roleview = new RoleView(rolePrincipal.getRole());
        List<UserView> rolePrincipalsView = userViewConverter.toUserView(rolePrincipal.getPrincipals(), cache, false);
        return new RolePrincipalsView(roleview, rolePrincipalsView);
    }

    @PUT
    @Timed
    @Path("principals")
    public void createOrUpdateGlobalRolePermission(final RolePrincipalsView rolePrincipalsView) {
        permissionChecker.check(EDIT_SECURITY);
        RolePrincipals rolePrincipals = rolePrincipalsView.toRolePrincipals();
        teamService.generateIdIfNecessary(rolePrincipals.getRole());
        Role role = dtoReader.readRoleAssignments(Collections.singletonList(rolePrincipals)).get(0);
        logger.debug(String.format("createOrUpdateGlobalRolePermission: role := %s / %s", role.getId(), role.getName()));

        Optional<Role> originalRole = getRole(role.getId());

        roleService.createOrUpdateRole(role);
        RolePrincipalEventPublisher.publishCreateOrUpdate(originalRole, role);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
    }

    @DELETE
    @Timed
    @Path("principals/{roleId:.*}")
    public void deleteGlobalRolePermission(@PathParam("roleId") String roleId) {
        permissionChecker.check(EDIT_SECURITY);
        Optional<Role> role = getRole(roleId);
        roleService.deleteById(roleId);
        RolePrincipalEventPublisher.publishDelete(role);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
    }

    private List<RolePrincipals> getRolePrincipalsList() {
        List<Role> multimap = roleService.readRoleAssignments();
        return dtoWriter.writeRoleAssignments(multimap);
    }

    private Optional<Role> getRole(final String roleId) {
        return getRolePrincipalsList().stream()
                .filter(r -> roleId.equals(r.getRole().getId()))
                .findFirst()
                .map(r -> dtoReader.readRoleAssignments(Collections.singletonList(r)).get(0));
    }
}
