package com.xebialabs.xlrelease.api.internal;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import jakarta.ws.rs.*;
import jakarta.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.rest.api.SecurityResource;
import com.xebialabs.deployit.engine.api.security.Role;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.views.RolePermissionsView;
import com.xebialabs.xlrelease.views.RolesPermissionsView;

import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getGlobalPermissions;

/**
 * The permissions assigned to a security role.
 */

@Path("/roles/permissions")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class RolePermissionsResource {

    private SecurityResource securityResource;

    private PermissionChecker permissionChecker;

    private XLReleaseEventBus xlReleaseEventBus;

    @Autowired
    public RolePermissionsResource(SecurityResource securityResource, PermissionChecker permissionChecker, XLReleaseEventBus xlReleaseEventBus) {
        this.securityResource = securityResource;
        this.permissionChecker = permissionChecker;
        this.xlReleaseEventBus = xlReleaseEventBus;
    }

    @GET
    @Timed
    @Path("global")
    public RolesPermissionsView getGlobalRolePermissions() {
        permissionChecker.check(EDIT_SECURITY);

        List<RolePermissions> permissions = securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), null, null, null);
        Map<String, List<RolePermissions>> permissionLookup = permissions.stream().collect(Collectors.groupingBy(role -> role.getRole().getId()));


        List<RolePermissions> roles = new ArrayList<>();

        for (RolePrincipals rolePrincipals : securityResource.readRolePrincipals(null, null, null)) {
            Role role = rolePrincipals.getRole();
            if (permissionLookup.containsKey(role.getId())) {
                roles.addAll(permissionLookup.get(role.getId()));
            } else {
                roles.add(new RolePermissions(role, new ArrayList<>()));
            }
        }

        return new RolesPermissionsView(roles, getGlobalPermissions());
    }

    @PUT
    @Timed
    @Path("global")
    public void setGlobalRolePermissions(List<RolePermissionsView> rolePermissionsViews) {
        this.permissionChecker.check(EDIT_SECURITY);

        List<RolePermissions> permissions = rolePermissionsViews.stream().map(RolePermissionsView::toRolePermissions).collect(Collectors.toList());
        securityResource.writeRolePermissions(GLOBAL_SECURITY_ALIAS(), permissions);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
    }
}
