package com.xebialabs.xlrelease.features.distributed

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.configuration.FeatureSettings
import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventBus}
import com.xebialabs.xlrelease.service.ConfigurationService
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringAwareBean
import grizzled.slf4j.Logging
import org.springframework.beans.BeansException
import org.springframework.context.ApplicationContext

import scala.jdk.CollectionConverters._

trait BaseDistributedFeature[T <: FeatureSettings] extends DistributedFeatureTypeCapture[T] with ScalaSpringAwareBean with Logging {
  // it is important that trait extends abstract base class so that mbassador bus would not detect same event handlers multiple times
  private val featureTypeString = Type.valueOf(featureType).toString
  private lazy val eventBus: EventBus = springBean[EventBus]
  private lazy val configurationService: ConfigurationService = springBean[ConfigurationService]

  private var _featureSettings: T = _

  @throws[BeansException]
  override def setApplicationContext(applicationContext: ApplicationContext): Unit = {
    super.setApplicationContext(applicationContext)
    eventBus.register(this)
  }

  @AsyncSubscribe
  def onFeatureCreate(event: DistributedConfigurationCreatedEvent): Unit = onSettingsCreateOrUpdate(event)

  @AsyncSubscribe
  def onFeatureUpdate(event: DistributedConfigurationUpdatedEvent): Unit = onSettingsCreateOrUpdate(event)

  def featureSettings: T = {
    if (_featureSettings == null) {
      _featureSettings = configurationService.getFeatureSettings(featureTypeString).asInstanceOf[T]
    }
    _featureSettings
  }

  private def onSettingsCreateOrUpdate(event: DistributedConfigurationEvents): Unit = {
    if (event.configurationType == featureTypeString) {
      val newSettings = configurationService.getFeatureSettings(featureTypeString).asInstanceOf[T]
      logger.trace(s"Received event: $event. Going to update feature settings.")
      updateFeatureSettings(newSettings)
    }
  }

  private def updateFeatureSettings(newSettings: T): Unit = {
    val pds = Type.valueOf(featureType).getDescriptor.getPropertyDescriptors.asScala
    // default implementation copies properties
    for (pd <- pds) {
      pd.set(featureSettings, pd.get(newSettings))
    }
  }
}
