package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.SSERepository
import com.xebialabs.xlrelease.user.User
import org.springframework.stereotype.Service

import javax.ws.rs.sse.SseEventSink

@Service
@EventListener
class SseService(sseRepository: SSERepository, broadcastService: BroadcastService) {

  def subscribeUserToSink(sink: SseEventSink): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.sseRepository.addUserToSink(username, sink)
  }

  def subscribeTopicToUser(topic: String): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.subscribeTopicToUser(topic, username)
  }

  def subscribeTopicToUser(topic: String, username: String): Unit = {
    this.broadcastService.broadcast(DistributedSubscribeSSEEvent(topic, username), publishEventOnSelf = true)
  }

  def unsubscribeTopicToUser(topic: String): Unit = {
    val username = User.AUTHENTICATED_USER.getName
    this.unsubscribeTopicToUser(topic, username)
  }

  def unsubscribeTopicToUser(topic: String, username: String): Unit = {
    this.broadcastService.broadcast(DistributedUnsubscribeSSEEvent(topic, username), publishEventOnSelf = true)
  }

  def unsubscribeAllUsers(topic: String): Unit = {
    this.broadcastService.broadcast(DistributedUnsubscribeAllSSEEvent(topic), publishEventOnSelf = true)
  }

  def sendEvent(topic: String, eventName: String, payload: String): Unit = {
    this.broadcastService.broadcast(DistributedSendSSEEvent(topic, eventName, payload), publishEventOnSelf = true)
  }

  def hasActiveUsers(topic: String): Boolean = this.sseRepository.getUsers(topic).nonEmpty

  @AsyncSubscribe
  def onEvent(event: DistributedSSEEvent): Unit = {
    event match {
      case DistributedSubscribeSSEEvent(topic, username) =>
        this.sseRepository.addTopicToUser(topic, username)
      case DistributedUnsubscribeSSEEvent(topic, username) =>
        this.sseRepository.removeTopicToUser(topic, username)
      case DistributedUnsubscribeAllSSEEvent(topic) =>
        this.sseRepository.removeAllUsersFromTopic(topic)
      case DistributedSendSSEEvent(topic, eventName, payload) =>
        this.sseRepository.sendEventToSink(topic, eventName, payload)
    }
  }
}
