package com.xebialabs.xlrelease.scheduler.events.handlers

import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.scheduler.RunnerRegistry
import com.xebialabs.xlrelease.scheduler.events.{JobRunnerCreatedOrUpdatedEvent, JobRunnerDeletedEvent, JobRunnerEvent}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import scala.util.{Failure, Success, Try}

@Service
@EventListener
class JobRunnerEventHandler(broadcastService: BroadcastService, runnerRegistry: RunnerRegistry) extends Logging {

  @Subscribe
  def onJobRunnerCreated(event: ConfigurationEvent): Unit = {
    val runnerEvent: Option[JobRunnerEvent] = event match {
      case ConfigurationCreatedEvent(conf: JobRunner) => Some(JobRunnerCreatedOrUpdatedEvent(conf))
      case ConfigurationUpdatedEvent(updated: JobRunner) => Some(JobRunnerCreatedOrUpdatedEvent(updated))
      case ConfigurationDeletedEvent(conf: JobRunner) => Some(JobRunnerDeletedEvent(conf))
      case ConfigurationCopiedEvent(original: JobRunner, _) => Some(JobRunnerCreatedOrUpdatedEvent(original))
      case _ => None //nothing
    }

    runnerEvent.foreach(event => broadcastService.broadcast(event, publishEventOnSelf = true))
  }

  @Subscribe
  def onJobRunnerEvent(event: JobRunnerEvent): Unit = {
      event match {
        case JobRunnerCreatedOrUpdatedEvent(jobRunner) =>
          if (jobRunner.isEnabled && jobRunner.capacity > 0 && jobRunner.isAvailable) {
            registerRunner(jobRunner)
          } else {
            unregisterRunner(jobRunner, deleteRunner = false)
          }
        case JobRunnerDeletedEvent(jobRunner) =>
          unregisterRunner(jobRunner, deleteRunner = true)
      }
  }

  private def registerRunner(jobRunner: JobRunner): Unit = {
    Try(runnerRegistry.registerJobRunner(jobRunner)) match {
      case Failure(ex) =>
        logger.error(s"Unable to register job runner [${jobRunner.getId}]", ex)
        jobRunner.stop()
      case Success(_) =>
        logger.trace(s"Successfully registered job runner [${jobRunner.getId}]. Going to start it.")
        jobRunner.start()
    }
  }

  private def unregisterRunner(jobRunner: JobRunner, deleteRunner: Boolean): Unit = {
    Try(runnerRegistry.unregisterJobRunner(jobRunner)) match {
      case Failure(ex) => logger.error(s"Unable to unregister job runner [${jobRunner.getId}]", ex)
      case Success(_) => logger.trace(s"Unregistered job runner [${jobRunner.getId}]")
    }
    // System should try to stop the job runner even if unregister fails
    if (deleteRunner) {
      jobRunner.delete()
    } else {
      jobRunner.stop()
    }
  }

}
