package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlplatform.cluster.XlCluster
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.distributed.events.DistributedXLReleaseEvent
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.service.DbCredentialsService._
import com.xebialabs.xlrelease.utils.{ConfigEntry, ConfigUtils, ResourceUtils}
import com.zaxxer.hikari.HikariDataSource
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component

import java.sql.DriverManager
import java.util.Properties
import scala.util.{Failure, Success, Try}

object DbCredentialsService {
  val databaseUserConfigPath = "xl.database.db-username"
  val databasePasswordConfigPath = "xl.database.db-password"

  val reportingUserConfigPath = "xl.reporting.db-username"
  val reportingPasswordConfigPath = "xl.reporting.db-password"

  def successMessage(dbName: String): String = s"Credentials update request for $dbName database received. " +
    s"Please check the logs or xl-release.conf on individual nodes for confirmation"

  def invalidCredentialsMessage(dbName: String, message: String): String = s"Invalid credentials for $dbName. Detailed message: $message"

  def invalidDbNameMessage(dbName: String): String = {
    val commaSeparatedDbs: String = XlrDatabase.values.foldLeft("") {
      (res: String, value: Enumeration#Value) => res + value + ", "
    }
    val dbs: String = "[" + commaSeparatedDbs.substring(0, commaSeparatedDbs.lastIndexOf(",")) + "]"
    s"Unknown database: $dbName. Please use one of $dbs"
  }
}

@Component
@EventListener
class DbCredentialsService(@Qualifier("rawRepositoryDataSource")
                           xlrRepositoryDataSource: HikariDataSource,
                           @Qualifier("rawReportingDataSource")
                           reportingDataSource: HikariDataSource,
                           broadcastService: BroadcastService)
  extends Logging {

  @AsyncSubscribe
  def onDbCredentialsChangeRequested(event: DbCredentialsChangeRequested): Unit = {
    updateDbCredentials(event.dbUsername, event.dbPassword, event.database)
  }

  def updateDbCredentials(username: String, password: String, database: XlrDatabase.Value): Unit = {
    ResourceUtils.findFirstInClassPath(XlrConfig.defaultConfigName) match {
      case Some(file) =>
        val configEntries = createConfigEntries(username, password, database)
        val updatedConfig = ConfigUtils.updateConfig(XlrConfig.defaultConfigName, configEntries)
        ConfigUtils.writeToFile(file, updatedConfig) match {
          case Success(_) =>
            XlrConfig.reloadConfig()
            updateDataSource(database, XlrConfig.getInstance)
          case Failure(ex) => logger.error(s"Unable to update DB credentials on $database", ex)
        }
      case None =>
        warn(s"Configuration file ${XlrConfig.defaultConfigName} cannot be found in file system. Updating the DB credentials aborted!")
    }
  }

  def validateAndPublishCredentialsChangeRequest(dbUsername: String, dbPassword: String, dbName: String): DbCredentialsChangeResult = {
    validator(dbName) match {
      case Some(validator) => validator.validate(dbUsername, dbPassword) match {
        case Success(_) =>
          val encryptedPassword = PasswordEncrypter.getInstance.encrypt(dbPassword)
          broadcastService.broadcast(DbCredentialsChangeRequested(dbUsername, encryptedPassword, XlrDatabase.withName(dbName)), publishEventOnSelf = true)
          DbCredentialsChangeResult(success = true, successMessage(dbName))
        case Failure(ex) =>
          logger.error("Invalid db credentials. Update did not happen.", ex)
          DbCredentialsChangeResult(success = false, invalidCredentialsMessage(dbName, ex.getMessage))
      }
      case None => DbCredentialsChangeResult(success = false, invalidDbNameMessage(dbName))
    }
  }

  private def updateDataSource(database: XlrDatabase.Value, config: XlrConfig): Unit = {
    database match {
      case XlrDatabase.REPOSITORY =>
        setNewCredentialsOn(xlrRepositoryDataSource, config.xlrRepositoryUsername, config.xlrRepositoryPassword) match {
          case Success(_) =>
            if (XlCluster.getXlClusterProvider.isDefined) {
              val clusterConfig = XlrConfig.getInstance.cluster
              XlCluster.getXlClusterProvider.get.doWithDatasource { dataSource =>
                setNewCredentialsOn(dataSource, clusterConfig.membership.datasource.username, clusterConfig.membership.datasource.password) match {
                  case Success(_) => logger.info("Cluster membership database credentials successfully updated")
                  case Failure(exception) => logger.warn("Error updating cluster membership database credentials", exception)
                }
              }
            }
            logger.info("Repository database credentials successfully updated")
          case Failure(exception) => logger.warn("Error updating repository DB credentials", exception)
        }
      case XlrDatabase.ARCHIVE =>
        setNewCredentialsOn(reportingDataSource, config.reportingUsername, config.reportingPassword) match {
          case Success(_) => logger.info("Archive database credentials successfully updated")
          case Failure(exception) => logger.warn("Error updating archive DB credentials", exception)
        }
    }
  }

  private def validator(dbNameString: String): Option[DbConnectingCredentialsValidator] = {
    Try {
      XlrDatabase.withName(dbNameString)
    } match {
      case Success(dbName) =>
        dbName match {
          case XlrDatabase.REPOSITORY => Some(new RepositoryCredentialsValidator(XlrConfig.getInstance.xlrRepositoryJdbcUrl))
          case XlrDatabase.ARCHIVE => Some(new ArchiveCredentialsValidator(XlrConfig.getInstance.reportingUrl))
        }
      case Failure(_) => None
    }
  }

  private def setNewCredentialsOn(dataSource: HikariDataSource, dbUsername: String, dbPassword: String): Try[Unit] = {
    Try {
      dataSource.getHikariPoolMXBean.suspendPool()
      dataSource.getHikariConfigMXBean.setUsername(dbUsername)
      dataSource.getHikariConfigMXBean.setPassword(dbPassword)
      dataSource.getHikariPoolMXBean.softEvictConnections()
      dataSource.getHikariPoolMXBean.resumePool()
    }
  }

  private def createConfigEntries(username: String, password: String, database: XlrDatabase.Value): Seq[ConfigEntry] = {
    database match {
      case XlrDatabase.REPOSITORY =>
        val usernameEntry = ConfigEntry(databaseUserConfigPath, username)
        val passwordEntry = ConfigEntry(databasePasswordConfigPath, password)
        Seq(usernameEntry, passwordEntry)
      case XlrDatabase.ARCHIVE =>
        val usernameEntry = ConfigEntry(reportingUserConfigPath, username)
        val passwordEntry = ConfigEntry(reportingPasswordConfigPath, password)
        Seq(usernameEntry, passwordEntry)
      case _ => throw new RuntimeException(s"Unable to update $database credentials")
    }
  }
}

case class DbCredentialsChangeRequested(dbUsername: String, dbPassword: String, database: XlrDatabase.Value) extends DistributedXLReleaseEvent

object XlrDatabase extends Enumeration {
  val REPOSITORY: XlrDatabase.Value = Value("repository")
  val ARCHIVE: XlrDatabase.Value = Value("archive")
}

class RepositoryCredentialsValidator(url: String) extends DbConnectingCredentialsValidator(url, "SELECT COUNT(*) FROM XLR_RELEASES")

class ArchiveCredentialsValidator(url: String) extends DbConnectingCredentialsValidator(url, "SELECT COUNT(*) FROM RELEASES")

class DbConnectingCredentialsValidator(url: String, query: String) {

  def validate(username: String, password: String): Try[_] = {
    val connectionProps = new Properties()
    connectionProps.put("user", username)
    connectionProps.put("password", password)

    Try {
      val connection = DriverManager.getConnection(url, connectionProps)
      val statement = connection.prepareStatement(query)
      statement.execute()
    }
  }

}

case class DbCredentialsChangeResult(success: Boolean, message: String)