package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.domain.Configuration
import com.xebialabs.xlrelease.script.EncryptionHelper.decrypt
import com.xebialabs.xlrelease.views.AutoconfigResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

trait ConfigurationAutoconfigService {
  def isAutoconfigure(configuration: Configuration): Boolean
  def autoconfigure(configuration: Configuration): AutoconfigResponse
}

@Service
class DefaultConfigurationAutoconfigService @Autowired()(autoconfigExecutors: JList[AutoconfigExecutor]) extends ConfigurationAutoconfigService {
  @Timed
  override def isAutoconfigure(configuration: Configuration): Boolean = {
    getExecutor(configuration).isDefined
  }

  @Timed
  override def autoconfigure(configuration: Configuration): AutoconfigResponse = {
    getExecutor(configuration) match {
      case Some(executor) =>
        decrypt(configuration)
        executor.autoconfigure(configuration)
      case None =>
        val msg = s"Configuration '${configuration.getTitle}' of type '${configuration.getType}' does not have auto-configuration logic implemented."
        new AutoconfigResponse(false, msg, null, null)
    }
  }

  def getExecutor(configuration: Configuration): Option[AutoconfigExecutor] = {
    autoconfigExecutors.asScala.find(_.supports(configuration))
  }
}

trait AutoconfigExecutor {
  def supports(configuration: Configuration): Boolean
  def autoconfigure(configuration: Configuration): AutoconfigResponse
}


