package com.xebialabs.xlrelease.repository.query

import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Dialect, MSSQLDialect}
import grizzled.slf4j.Logging
import org.springframework.data.domain.Sort.Order
import org.springframework.data.domain.{Pageable, Sort}
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate

import scala.collection.mutable
import scala.jdk.StreamConverters._

trait FiltersQueryBuilder[F, R] {
  val NL = System.lineSeparator

  type SelfType = FiltersQueryBuilder[F, R]
  var withoutPaging = false
  var pageable: Pageable = _

  def from(filter: F): SelfType

  def withPageable(pageable: Pageable): SelfType = {
    this.pageable = pageable
    this
  }

  def withSortParameters(sorts: (String, String)*): SelfType

  def withPermissions(permission: Seq[Permission], principals: Iterable[String], roleIds: Iterable[String]): SelfType = this

  def turnOffPaging(): SelfType = {
    this.withoutPaging = true
    this
  }

  def build(): PageableQuery[R]
}

trait FilterQueryBuilderSupport[F, R] extends LimitOffset with Logging {
  self: FiltersQueryBuilder[F, R] =>
  val dialect: Dialect
  val namedTemplate: NamedParameterJdbcTemplate
  val queryParams: mutable.Map[String, Any] = mutable.HashMap()
  val joinClauses: mutable.Buffer[String] = mutable.Buffer()
  val whereClauses: mutable.Buffer[String] = mutable.Buffer()
  val orderClauses: mutable.Buffer[String] = mutable.Buffer()
  val sortOrderMapping: mutable.Map[String, String] = mutable.HashMap()

  override def withSortParameters(sorts: (String, String)*): SelfType = {
    this.sortOrderMapping ++= sorts
    this
  }

  private def escapeSquareBrackets(value: String): String = value.replaceAll("\\[", "\\\\[").replaceAll("\\]", "\\\\]")

  private def escapeStatement: String = "escape '\\'"

  def like(column: String, key: String, value: String): SelfType = {
    if (value != null && !value.isEmpty) {
      dialect match {
        case MSSQLDialect(_) =>
          whereClauses += s"LOWER($column) LIKE :$key $escapeStatement"
          queryParams += key -> s"%${escapeSquareBrackets(value.toLowerCase())}%"
        case _ =>
          whereClauses += s"LOWER($column) LIKE :$key"
          queryParams += key -> s"%${value.toLowerCase()}%"
      }
    }
    this
  }

  def joinClause: String = {
    if (joinClauses.isEmpty) {
      ""
    } else {
      joinClauses.mkString(NL)
    }
  }

  def equal(column: String, key: String, value: String): SelfType = {
    if (value != null && !value.isEmpty) {
      whereClauses += s"$column = :$key"
      queryParams += key -> value
    }
    this
  }

  def whereClause: String = {
    if (whereClauses.isEmpty) {
      ""
    } else {
      s"WHERE ${whereClauses.mkString(" AND ")}"
    }
  }

  def orderClause: String = {
    if (orderClauses.isEmpty) {
      ""
    } else {
      s"ORDER BY ${orderClauses.mkString(", ")}"
    }
  }

  def pageableQuery(query: String): String = {
    if (!withoutPaging && pageable != null && !pageable.isUnpaged) {
      addLimitAndOffset(query, pageable.getPageSize, pageable.getOffset)
    } else {
      query
    }
  }

  def buildSortOrderClause(defaultOrderColumn: String): Unit = {
    val defaultOrder = Order.asc(defaultOrderColumn)
    val defaultSort = Sort.by(defaultOrder)
    val orders: Seq[Order] = if (pageable != null) {
      pageable.getSortOr(defaultSort).get().toScala(Seq)
    } else {
      Seq(defaultOrder)
    }
    val (validOrders, invalidOrders) = orders.partition(order => sortOrderMapping.contains(order.getProperty))
    if (invalidOrders.nonEmpty) logger.warn(s"Ignoring unexpected sort order parameters: $invalidOrders")
    validOrders.map(order => s"${sortOrderMapping.getOrElse(order.getProperty, s"could not find mapping for ${order.getProperty}")} ${order.getDirection}")
      .foreach(o => orderClauses += o)
  }
}
