package com.xebialabs.xlrelease.actors.initializer

import akka.actor.{ActorRef, PoisonPill, Props}
import akka.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings}
import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlrelease.actors
import com.xebialabs.xlrelease.actors.{ActorServiceHolder, ActorSystemHolder, ReleaseCleanupActor}
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.springframework.context.annotation.Profile
import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

trait ReleaseCleanupInitializer extends ActorInitializer with Logging {
  def props(xlrConfig: XlrConfig, actorServiceHolder: ActorServiceHolder): Props = {
    Props(new actors.ReleaseCleanupActor(actorServiceHolder.releaseCleanupScheduleService, xlrConfig))
  }
}

@Component
@Order(2)
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredReleaseCleanupInitializer(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ReleaseCleanupInitializer {

  lazy val actor: ActorRef = systemHolder.actorSystem.actorOf(props(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] release cleanup scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing non-clustered release cleanup actor...")
      actor
    }
  }
}

@Component
@Order(2)
@Profile(Array(FULL))
class ClusteredReleaseCleanupInitializer(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ReleaseCleanupInitializer with Logging {

  lazy val actor: ActorRef = clusteredReleaseCleanupActor(props(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] release cleanup scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing clustered release cleanup actor...")
      actor
    }
  }

  private[initializer] def clusteredReleaseCleanupActor(props: Props): ActorRef = {
    systemHolder.actorSystem.actorOf(ClusterSingletonManager.props(
      singletonProps = props,
      terminationMessage = PoisonPill,
      settings = ClusterSingletonManagerSettings(systemHolder.actorSystem)),
      name = ReleaseCleanupActor.name)
  }
}

