package com.xebialabs.xlrelease.actors

import com.github.benmanes.caffeine.cache.{Cache, Caffeine}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.utils.ReleaseSoftReference
import grizzled.slf4j.Logging

import java.util.concurrent.TimeUnit

object ReleaseActorCache extends Logging {
  private lazy val cacheEnabled = XlrConfig.getInstance.features.releaseActors.releaseCacheEnabled
  private lazy val cache: Cache[String, ReleaseSoftReference] = Caffeine.newBuilder()
    .initialCapacity(2000)
    .maximumSize(3000)
    .expireAfterAccess(5, TimeUnit.MINUTES)
    .recordStats()
    .build()

  def get(id: String)(supplier: java.util.function.Supplier[Release]): Release = {
    if (cacheEnabled) {
      val shortId = releaseId2ActorName(id)
      val ref = cache.get(shortId, _ => {
        val release = supplier.get();
        logger.debug(s"miss ${id}")
        new ReleaseSoftReference(release, supplier)
      })
      ref.get()
    } else {
      supplier.get()
    }
  }

  def getIfPresent(id: String): Option[Release] = {
    Option(cache.getIfPresent(id)).flatMap(ref => Option(ref.getIfPresent()))
  }

  def remove(id: String): Option[Release] = {
    if (null != id && cacheEnabled) {
      val shortId = releaseId2ActorName(id)
      val cachedReleaseRef = Option(cache.getIfPresent(shortId))
      cache.invalidate(shortId)
      cachedReleaseRef.flatMap(ref => Option(ref.get()))
    } else {
      None
    }
  }

}
