package com.xebialabs.xlrelease;

import java.io.*;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.crypto.SecretKey;
import com.typesafe.config.ConfigFactory;

import com.xebialabs.deployit.ServerConfigFile;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.booter.local.utils.Closeables;
import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.setup.FileCopier;
import com.xebialabs.deployit.setup.UpgradeHelper;
import com.xebialabs.deployit.util.DeployitKeyStore;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.overthere.util.OverthereUtils;
import com.xebialabs.xlplatform.io.FolderChecksum;
import com.xebialabs.xlrelease.config.XlrConfig;

import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_HTTPS_PORT;
import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_HTTP_BIND_ADDRESS;
import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_HTTP_PORT;
import static com.xebialabs.deployit.setup.SetupHelperMethods.EXITSETUP_KEYWORD;
import static com.xebialabs.deployit.setup.SetupHelperMethods.getBooleanResponse;
import static com.xebialabs.deployit.setup.SetupHelperMethods.getConfirmedPassword;
import static com.xebialabs.deployit.setup.SetupHelperMethods.getStringResponse;
import static com.xebialabs.deployit.setup.SetupHelperMethods.getValidIntegerResponse;
import static com.xebialabs.deployit.setup.SetupHelperMethods.printEmptyLine;
import static com.xebialabs.overthere.util.OverthereUtils.checkState;
import static java.util.Comparator.reverseOrder;
import static scala.jdk.javaapi.CollectionConverters.asJava;

public class XLReleaseSetup {

    private final String serverName;
    private final ServerConfigFile serverConfigFile;
    private final XLReleaseServerLaunchOptions launchOptions;

    private ServerConfiguration configuration;

    // Wizard options
    private boolean askServerUrl = true;

    // Setup options
    private boolean generateKeyStore;
    private boolean generatePasswordEncryptionKey;
    private String passwordEncryptionKeyPassword;

    // Setup results

    public XLReleaseSetup(String serverName, ServerConfigFile serverConfigFile, XLReleaseServerLaunchOptions launchOptions, ServerConfiguration serverConfiguration) {
        this.serverName = serverName;
        this.serverConfigFile = serverConfigFile;
        this.launchOptions = launchOptions;
        this.configuration = serverConfiguration;
    }

    public void run() {
        final String previousInstallation = launchOptions.getPreviousInstallation();
        if (Strings.isNotBlank(previousInstallation)) {
            runUpgrade(previousInstallation);
        } else if (launchOptions.isReinitialize()) {
            reinitialize();
        } else {
            runComplete();
        }
    }

    private void runUpgrade(final String previousInstallation) {
        if (!launchOptions.isDoSetup()) {
            throw new RuntimeException("Option `-previous-installation` is only supported in setup mode (with `-setup` option)");
        } else {
            final UpgradeHelper upgradeHelper = new UpgradeHelper(previousInstallation, launchOptions.isForceUpgrades(), new FileCopier());

            final Path previousInstallationPath = new File(previousInstallation).toPath();
            final File confFolder = new File(String.format("%s/conf", previousInstallation));
            final List<String> configFiles = asJava(FolderChecksum.listAllFilesSorted(confFolder))
                    .stream()
                    .filter(file -> !file.getName().equalsIgnoreCase("script.policy"))
                    .filter(file -> !file.getName().equalsIgnoreCase("xlr-wrapper-linux.conf"))
                    .filter(file -> !file.getName().equalsIgnoreCase("xlr-wrapper-win.conf"))
                    .map(file -> previousInstallationPath.relativize(file.toPath()).toString()).collect(Collectors.toList());
            final XlrConfig xlrConfig = new XlrConfig(ConfigFactory.parseFile(Paths.get(previousInstallation, "conf", "xl-release.conf").toFile())
                    .withFallback(XlrConfig.bootConfig().rootConfig()));
            final String reportFolder = xlrConfig.reportingEngineLocation();
            final List<String> dirs = new ArrayList<>();
            if (!Paths.get(reportFolder).isAbsolute()) {
                dirs.add(reportFolder);
            }
            dirs.add("ext");
            upgradeHelper.copyData(configFiles, dirs, Arrays.asList("conf/script.policy", "conf/xlr-wrapper-linux.conf", "conf/xlr-wrapper-win.conf"));
        }
    }

    private void reinitialize() {
        if (!launchOptions.isForce()) {
            System.out.println("The -reinitialize will only work with a default repository setup and not when Digital.ai Release is configured with an external database.");
            System.out.println("Are you sure you want to continue (yes or no)?");
            if (!getBooleanResponse(false)) {
                return;
            }
        }

        // Load password encryption keys
        String repositoryKeystorePassword = launchOptions.getRepositoryKeystorePassword();
        SecretKey passwordEncryptionKey = null;
        if (null != repositoryKeystorePassword) {
            DeployitKeyStore.generateRandomKeyStore(new File("conf"), repositoryKeystorePassword);
            passwordEncryptionKey = DeployitKeyStore.getPasswordEncryptionKey();
        } else {
            passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(repositoryKeystorePassword);
        }
        PasswordEncrypter.init(passwordEncryptionKey);

        useDefaultValues(launchOptions.getDefaultsFile());
        serverConfigFile.writeConfiguration(configuration);
        configuration = serverConfigFile.loadConfig();
        initializeRepository();
        System.out.println("Finished reinitialization.");
    }

    private void useDefaultValues(String defaultsFile) {
        int httpPort = configuration.getHttpPort();
        configuration.setDefaults();
        configuration.setHttpPort(httpPort);
        if (defaultsFile != null) {
            File file = new File(defaultsFile);
            checkState(file.exists(), "The given file [%s] with defaults does not exist.", defaultsFile);
            Properties properties = new Properties();
            FileInputStream inStream = null;
            try {
                inStream = new FileInputStream(file);
                properties.load(inStream);
            } catch (IOException e) {
                throw new RuntimeException("Cannot load [" + file + "] to read defaults from.");
            } finally {
                Closeables.closeQuietly(inStream);
            }
            configuration.load(properties);
        }
    }

    private void runComplete() {
        System.out.println("\nWelcome to the " + serverName + " setup.");
        System.out.println("You can always exit by typing '" + EXITSETUP_KEYWORD + "'.");
        System.out.println("To re-run this setup and make changes to the " + serverName +
                " configuration you can run run.cmd -setup on Windows or run.sh -setup on Unix.");

        boolean editingExistingConfiguration = false;
        if (serverConfigFile.exists()) {
            if (askDoYouWantToEditTheExistingConfiguration()) {
                SecretKey passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword());
                PasswordEncrypter.init(passwordEncryptionKey);
                serverConfigFile.readIntoConfiguration(configuration);
                editingExistingConfiguration = true;
            }
        }

        if (askToUseSimpleSetup()) {
            if (!editingExistingConfiguration) {
                askAdminPassword();
                askToCreatePasswordEncryptionKey();
            } else {
                upgradeConfigurationWithDefaultValues();
            }
        } else {
            if (!editingExistingConfiguration) {
                askAdminPassword();
                askToCreatePasswordEncryptionKey();
            }

            if (askToEnableSsl()) {
                if (askToGenerateKeys()) {
                    setDefaultKeyStoreSettings();
                } else {
                    askKeyStoreSettings();
                }

                if (askToEnableMutualSsl()) {
                    askTrustStoreSettings();
                }
            }

            askHttpBindAddressForJetty();
            askHttpPortForJetty();
            askWebContextRoot();
            askServerUrl();
            askMinumAmountOfThreads();
            askMaximumAmountOfThreads();
        }

        if (reviewAndConfirm()) {
            SecretKey passwordEncryptionKey = null;
            if (generatePasswordEncryptionKey) {
                DeployitKeyStore.generateRandomKeyStore(new File("conf"), passwordEncryptionKeyPassword);
                passwordEncryptionKey = DeployitKeyStore.getPasswordEncryptionKey();
            } else {
                // Default key store
                passwordEncryptionKey = DeployitKeys.getPasswordEncryptionKey(launchOptions.getRepositoryKeystorePassword());
            }
            PasswordEncrypter.init(passwordEncryptionKey);

            System.out.println("Saving to " + serverConfigFile);
            serverConfigFile.writeConfiguration(configuration);
            System.out.println("Configuration saved.");

            if (generateKeyStore) {
                generateKeyStore();
            }

            // Load the new configuration
            configuration = serverConfigFile.loadConfig();

            System.out.println("You can now start your " + serverName + " by executing the command run.cmd on Windows or run.sh on Unix.");
            System.out.println("Note: If your " + serverName + " is running please restart it.");
            System.out.println("Finished setup.");
        } else {
            System.out.println("Aborting setup.");
        }
    }

    private boolean reviewAndConfirm() {
        printEmptyLine();
        System.out.println("Do you agree with the following settings for " + serverName + " and would you like to save them?");
        System.out.println("Changes will be saved in " + serverConfigFile.getName());
        System.out.println("\tSSL will be " + (configuration.isSsl() ? "enabled" : "disabled"));
        if (configuration.isSsl()) {
            if (generateKeyStore) {
                System.out.println("\tKeystore will be generated");
            } else {
                System.out.println("\tKeystore path is " + configuration.getKeyStorePath());
                System.out.println("\tKeystore password is " + configuration.getKeyStorePassword());
                System.out.println("\tKeystore key password is " + configuration.getKeyStoreKeyPassword());
            }
        }
        System.out.println("\tHTTP bind address is " + configuration.getHttpBindAddress());
        System.out.println("\tHTTP port is " + configuration.getHttpPort());
        System.out.println("\tContext root is " + configuration.getWebContextRoot());
        if (askServerUrl) {
            System.out.println("\tPublic Server URL is " + configuration.getServerUrl());
        }
        System.out.println("\tHTTP server will use a minimum of " + configuration.getMinThreads() + " and a maximum of " + configuration.getMaxThreads() + " threads");

        if (generatePasswordEncryptionKey) {
            System.out.println("\tPassword encryption key will be generated.");
        }

        return getBooleanResponse(true);
    }


    private boolean askDoYouWantToEditTheExistingConfiguration() {
        boolean useOldConfig;
        printEmptyLine();
        System.out.println("An existing " + serverName + " configuration was found. Do you want to edit it?");
        System.out.println("Options are yes or no. (selecting no will create an empty configuration)");
        useOldConfig = getBooleanResponse(true);
        if (useOldConfig)
            System.out.println("Editing the exisiting configuration.");
        else
            System.out.println("Starting configuration from scratch.");
        return useOldConfig;
    }

    private void upgradeConfigurationWithDefaultValues() {
        if (configuration.getMinThreads() == 0) {
            configuration.setMinThreads(ServerConfiguration.DEFAULT_MIN_THREADS);
        }
        if (configuration.getMaxThreads() == 0) {
            configuration.setMaxThreads(ServerConfiguration.DEFAULT_MAX_THREADS);
        }
    }

    private void askToCreatePasswordEncryptionKey() {
        printEmptyLine();
        System.out.println("The password encryption key protects the passwords stored in the repository. ");
        System.out.println("Do you want to generate a new password encryption key?");
        System.out.println("Options are yes or no.");

        generatePasswordEncryptionKey = getBooleanResponse(true);
        if (generatePasswordEncryptionKey) {
            System.out.println("The password encryption key is optionally secured by a password.");
            System.out.println("Please enter the password you wish to use. (Use an empty password to avoid a password prompt when starting " + serverName + ".)");
            passwordEncryptionKeyPassword = getConfirmedPassword();
        }

    }


    private static boolean askToUseSimpleSetup() {
        printEmptyLine();
        System.out.println("Do you want to use the simple setup?");
        System.out.println("Default values are used for all properties. To make changes to the default properties, please answer no.");
        System.out.println("Options are yes or no.");
        return getBooleanResponse(true);
    }

    private void askAdminPassword() {
        printEmptyLine();
        System.out.println("Please enter the admin password you wish to use for " + serverName);

        String password = getConfirmedPassword();

        if (Strings.isEmpty(password)) {
            System.out.println("Using default admin password.");
            configuration.setAdminPassword(ServerConfiguration.DEFAULT_ADMIN_PASSWORD);
        } else {
            configuration.setAdminPassword(password);
        }
    }

    private void askWebContextRoot() {
        printEmptyLine();
        System.out.println("Enter the web context root where " + serverName + " will run");
        String contextRoot = getStringResponse(ServerConfiguration.DEFAULT_WEB_CONTEXT_ROOT);
        if (!contextRoot.startsWith("/")) {
            contextRoot = "/" + contextRoot;
        }
        configuration.setWebContextRoot(contextRoot);
    }

    private void askServerUrl() {
        if (!askServerUrl) {
            return;
        }

        printEmptyLine();
        System.out.println("Enter the public URL to access " + serverName);
        String derivedUrl = configuration.getDerivedServerUrl();
        String serverUrl = getStringResponse(derivedUrl);

        // Only store if different
        if (!derivedUrl.equals(serverUrl)) {
            configuration.setServerUrl(serverUrl);
        }
    }

    private boolean askToEnableSsl() {
        printEmptyLine();
        System.out.println("Would you like to enable SSL?");
        System.out.println("Options are yes or no.");
        boolean suppliedSsl = configuration.isNewConfiguration() || configuration.isSsl();
        configuration.setSsl(getBooleanResponse(suppliedSsl));
        return configuration.isSsl();
    }

    private boolean askToGenerateKeys() {
        printEmptyLine();
        System.out.println("Would you like to generate a keystore with a self-signed certificate for you?");
        System.out.println("N.B.: Self-signed certificates do not work correctly with some versions of the Flash Player and some browsers!");
        System.out.println("Options are yes or no.");
        boolean suppliedGenerateKeys = configuration.isNewConfiguration() || generateKeyStore;
        generateKeyStore = getBooleanResponse(suppliedGenerateKeys);
        return generateKeyStore;
    }

    private void setDefaultKeyStoreSettings() {
        configuration.setKeyStorePath("conf/keystore.p12");
        configuration.setKeyStorePassword("keystoresecret");
        configuration.setKeyStoreKeyPassword("keystoresecret");
    }

    private void askKeyStoreSettings() {
        printEmptyLine();
        System.out.println("What is the path to the keystore?");
        String suppliedKeyStorePath = configuration.isNewConfiguration() ? "" : configuration.getKeyStorePath();
        configuration.setKeyStorePath(getStringResponse(suppliedKeyStorePath));

        printEmptyLine();
        System.out.println("What is the password to the keystore?");
        String suppliedKeyStorePassword = configuration.isNewConfiguration() ? "" : configuration.getKeyStorePassword();
        configuration.setKeyStorePassword(getStringResponse(suppliedKeyStorePassword));

        printEmptyLine();
        System.out.println("What is the password to the key in the keystore?");
        String suppliedKeyStoreKeyPassword = configuration.isNewConfiguration() ? "" : configuration.getKeyStoreKeyPassword();
        configuration.setKeyStoreKeyPassword(getStringResponse(suppliedKeyStoreKeyPassword));
    }

    private boolean askToEnableMutualSsl() {
        printEmptyLine();
        System.out.println("Would you like to enable mutual SSL?");
        System.out.println("Options are yes or no.");
        boolean suppliedMutualSsl = configuration.isNewConfiguration() || configuration.isMutualSsl();
        configuration.setMutualSsl(getBooleanResponse(suppliedMutualSsl));
        return configuration.isMutualSsl();
    }

    private void askTrustStoreSettings() {
        printEmptyLine();
        System.out.println("What is the path to the truststore?");
        String suppliedTrustStorePath = configuration.isNewConfiguration() ? "" : configuration.getTrustStorePath();
        configuration.setTrustStorePath(getStringResponse(suppliedTrustStorePath));

        printEmptyLine();
        System.out.println("What is the password to the truststore?");
        String suppliedTrustStorePassword = configuration.isNewConfiguration() ? "" : configuration.getTrustStorePassword();
        configuration.setTrustStorePassword(getStringResponse(suppliedTrustStorePassword));
    }

    private void askHttpBindAddressForJetty() {
        printEmptyLine();
        System.out.println("What http bind address would you like the server to listen on?");
        String defaultAddress = Strings.isEmpty(configuration.getHttpBindAddress()) ? DEFAULT_HTTP_BIND_ADDRESS : configuration.getHttpBindAddress();
        configuration.setHttpBindAddress(getStringResponse(defaultAddress));
    }

    private void askHttpPortForJetty() {
        printEmptyLine();
        System.out.println("What http port number would you like the server to listen on?");
        int defaultPort = configuration.isSsl() ? DEFAULT_HTTPS_PORT : DEFAULT_HTTP_PORT;
        int suppliedPort = (configuration.getHttpPort() == 0) ? defaultPort : configuration.getHttpPort();
        configuration.setHttpPort(getValidIntegerResponse(suppliedPort));
    }

    private void askMinumAmountOfThreads() {
        printEmptyLine();
        System.out.println("Enter the minimum number of threads the HTTP server should use (recommended: 3 per client, so 30 for 10 concurrent users)");
        int minThreads = ServerConfiguration.DEFAULT_MIN_THREADS;
        int suppliedMinThreads = (configuration.getMinThreads() == 0) ? minThreads : configuration.getMinThreads();
        configuration.setMinThreads(getValidIntegerResponse(suppliedMinThreads));
    }

    private void askMaximumAmountOfThreads() {
        printEmptyLine();
        System.out.println("Enter the maximum number of threads the HTTP server should use (recommended :3 per client, so 150 for 50 concurrent users)");
        int maxThreads = ServerConfiguration.DEFAULT_MAX_THREADS;
        int suppliedMaxThreads = (configuration.getMaxThreads() == 0) ? maxThreads : configuration.getMaxThreads();
        configuration.setMaxThreads(getValidIntegerResponse(suppliedMaxThreads));
    }

    private void generateKeyStore() {
        String keyAlgorithm = "RSA";
        String keyStoreKeyAlias = "jetty";
        String dname = "CN=localhost,O=" + serverName + ",C=NL";

        try {
            File keyStoreFile = new File(configuration.getKeyStorePath());
            if (keyStoreFile.exists()) {
                System.out.println("Existing keystore " + configuration.getKeyStorePath() + " deleted.");
                keyStoreFile.delete();
            }
            System.out.println("Generating keystore...");
            String[] keytoolArgs = new String[]{"keytool",
                    "-genkey",
                    "-keyalg", keyAlgorithm,
                    "-keystore", configuration.getKeyStorePath(),
                    "-storepass", configuration.getKeyStorePassword(),
                    "-alias", keyStoreKeyAlias,
                    "-keypass", configuration.getKeyStoreKeyPassword(),
                    "-validity", "366",
                    "-dname", dname,
                    "-deststoretype", "pkcs12"};
            Process p = Runtime.getRuntime().exec(keytoolArgs);

            Thread outputThread = new Thread(new InputStreamToOutputStream(p.getInputStream(), System.out));
            outputThread.start();
            Thread errorThread = new Thread(new InputStreamToOutputStream(p.getErrorStream(), System.err));
            errorThread.start();

            int exitValue;
            try {
                exitValue = p.waitFor();
            } catch (InterruptedException exc) {
                throw new IOException(exc.toString());
            }
            if (exitValue != 0) {
                throw new IOException("keytool exited with status code " + exitValue);
            }
            System.out.println("Keystore generated.");
        } catch (IOException exc) {
            System.err.println("WARNING: Could not generate keystore " + configuration.getKeyStorePath() + ": " + exc.toString());
        }
    }

    private void initializeRepository() {
        deleteDatabase("repository");
        deleteDatabase("archive");
    }

    @SuppressWarnings("ResultOfMethodCallIgnored")
    private void deleteDatabase(final String database) {
        Path repositoryDbPath = Paths.get(database);
        if (Files.exists(repositoryDbPath)) {
            try (Stream<Path> walk = Files.walk(repositoryDbPath)) {
                walk.map(Path::toFile)
                        .sorted(reverseOrder())
                        .forEach(File::delete);
            } catch (IOException e) {
                System.err.println("ERROR: Could not delete " + database + " database. " + e.getMessage());
            }
        }
    }

    private static class InputStreamToOutputStream implements Runnable {
        private InputStream in;
        private OutputStream out;

        public InputStreamToOutputStream(InputStream in, OutputStream out) {
            this.in = in;
            this.out = out;
        }

        public void run() {
            OverthereUtils.write(in, out);
        }
    }
}
