package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.{IncorrectArgumentException, checkArgument}
import com.xebialabs.xlrelease.domain.events.{FolderVariableCreatedEvent, FolderVariableDeletedEvent, FolderVariableUpdatedEvent}
import com.xebialabs.xlrelease.domain.variables.{FolderVariables, PasswordStringVariable, Variable}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.IdMatchers.FolderId
import com.xebialabs.xlrelease.repository.{FolderRepository, FolderVariableRepository}
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeeded
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

@Service
class FolderVariableService @Autowired()(folderRepository: FolderRepository,
                                         folderVariableRepository: FolderVariableRepository,
                                         eventBus: XLReleaseEventBus) {

  @Timed
  def findById(variableId: String): Variable = {
    folderVariableRepository.findById(variableId)
  }

  @Timed
  def createFolderVariable(variable: Variable): Variable = {
    variable.getFolderId match {
      case FolderId(folderId) =>
        checkArgument(folderRepository.exists(folderId), s"Folder with id='$folderId' doesn't exist")
        variable.checkFolderVariableValidity()
      case invalidId => throw new IncorrectArgumentException(s"You can't add variable with id='$invalidId'")
    }
    if (variable.isPassword) {
      variable.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
    }
    replacePasswordPropertiesInCiIfNeeded(None, variable)

    val created = folderVariableRepository.create(variable)
    eventBus.publish(FolderVariableCreatedEvent(created, null))
    created
  }

  @Timed
  def getAllFromImmediateParent(variableParentId: String): FolderVariables =
    new FolderVariables(variableParentId, folderVariableRepository.getAllFromParent(variableParentId))

  @Timed
  def getAllFromAncestry(variableParentId: String): FolderVariables =
    new FolderVariables(variableParentId, folderVariableRepository.getAllFromAncestry(variableParentId))

  @Timed
  def updateFolderVariable(variable: Variable): Variable = {
    variable.checkFolderVariableValidity()
    val current = folderVariableRepository.findById(variable.getId)
    checkSameType(current, variable)
    checkArgument(current.getFolderId == variable.getFolderId, "Cannot change folderId")

    if (variable.isPassword) {
      variable.asInstanceOf[PasswordStringVariable].checkExternalPasswordVariable()
    }
    replacePasswordPropertiesInCiIfNeeded(Some(current), variable)

    if (variable.getValueProvider != null) {
      variable.getValueProvider.setVariable(variable)
    }

    val updated = folderVariableRepository.update(variable)
    eventBus.publish(FolderVariableUpdatedEvent(current, updated, null))
    updated
  }

  @Timed
  def deleteFolderVariable(variableId: String): Unit = {
    val variable = findById(variableId)
    folderVariableRepository.delete(variableId)
    eventBus.publish(FolderVariableDeletedEvent(variable, null))
  }

  @Timed
  def findByKey(variableKey: String, variableHolderId: String): Variable = {
    folderVariableRepository.findByKey(variableKey, variableHolderId)
  }

  private def checkSameType(current: Variable, updated: Variable): Unit = {
    if (!current.getType.equals(updated.getType)) {
      throw new IllegalArgumentException(s"Cannot change type of variable ${updated.getId} from ${current.getType} to ${updated.getType}")
    }
  }
}
