package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.service.BackgroundJobAction.JobExecutionContext
import grizzled.slf4j.Logging
import org.joda.time.{Duration, LocalDateTime}

import java.util.concurrent.atomic.AtomicBoolean
import scala.annotation.tailrec
import scala.util.{Failure, Success, Try}

trait BackgroundJobAction[IdentifierType <: AnyRef] extends Logging {
  protected def queryIdentifiers(jobStartTime: LocalDateTime, jobQueryPeriodInHours: Int, pageSize: Int): List[IdentifierType]

  protected def processIdentifier(identifier: IdentifierType): Unit

  def execute(context: JobExecutionContext): Unit = {
    var identifiers: List[IdentifierType] = queryIdentifiers(context.jobStartTime, context.periodInHours, context.queryPageSize)
    while (!context.isCancelled && !context.ranTooLong() && (processIdentifiers(context, identifiers) || identifiers.nonEmpty)) {
      identifiers = queryIdentifiers(context.jobStartTime, context.periodInHours, context.queryPageSize)
    }
  }

  @tailrec
  private final def processIdentifiers(context: JobExecutionContext, identifiers: List[IdentifierType]): Boolean = {
    identifiers match {
      case Nil => false
      case identifier :: next =>
        if (context.ranTooLong() || context.isCancelled) {
          false
        } else {
          Try(processIdentifier(identifier)) match {
            case Success(_) =>
            case Failure(ex) => logger.warn(s"processing item ${identifier.toString} failed: ${ex.toString}")
          }
          context.sleepIfNeeded()
          processIdentifiers(context, next)
        }
    }
  }

}


object BackgroundJobAction {
  case class JobExecutionContext(cancelRequested: AtomicBoolean,
                                 jobDisplayName: String,
                                 maxSecondsForJob: Int,
                                 sleepSecondsBetweenItems: Int,
                                 jobStartTime: LocalDateTime,
                                 periodInHours: Int,
                                 queryPageSize: Int) extends Logging {
    def getDurationMillis(): Long = {
      val duration = new Duration(jobStartTime.toDateTime, LocalDateTime.now().toDateTime)
      duration.getMillis
    }

    def isCancelled: Boolean = {
      cancelRequested.get()
    }

    def ranTooLong(): Boolean = {
      if (maxSecondsForJob > 0) {
        val interrupt = getDurationMillis() > (maxSecondsForJob * 1000)
        if (interrupt) {
          logger.debug(s"$jobDisplayName paused because it has exceeded $maxSecondsForJob seconds.")
        }
        interrupt
      } else {
        false
      }
    }

    def sleepIfNeeded(): Unit = {
      if (sleepSecondsBetweenItems > 0) {
        logger.debug(s"Sleeping for ${sleepSecondsBetweenItems} seconds before processing the next item")
        try {
          Thread.sleep(sleepSecondsBetweenItems * 1000)
        } catch {
          case _: InterruptedException =>
            logger.warn(s"Sleeping was interrupted")
        }
      }
    }
  }
}