package com.xebialabs.deployit.plumbing.export;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.core.xml.PasswordEncryptingCiConverter;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.serialization.ConfigurationItemReferenceResolver;
import com.xebialabs.deployit.plumbing.serialization.PortableConfigurationReference;
import com.xebialabs.deployit.plumbing.serialization.ResolutionContext;
import com.xebialabs.xlrelease.domain.BaseConfiguration;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xltype.serialization.CiReader;
import com.xebialabs.xltype.serialization.CiWriter;

import scala.Option;

/**
 * A PasswordEncryptingCiConverter extension that doesn't throw errors when reading unresolved CIs or passwords encrypted with a different server key.
 * CI references are left empty if not found.
 * Passwords are left with a {@link UnresolvedReferencesConfigurationItemConverter#DECRYPTION_FAILED_VALUE} String.
 * Custom configurations are written with title.
 */
public class UnresolvedReferencesConfigurationItemConverter extends PasswordEncryptingCiConverter implements ConfigurationItemReferenceResolver {
    public static final String DECRYPTION_FAILED_VALUE = UnresolvedReferencesConfigurationItemConverter.class.getName() + ".DECRYPTION_FAILED_VALUE";

    private ResolutionContext resolutionContext;
    private ConfigurationRepository configurationRepository;

    public UnresolvedReferencesConfigurationItemConverter(ConfigurationRepository configurationRepository, ResolutionContext resolutionContext) {
        this.configurationRepository = configurationRepository;
        this.resolutionContext = resolutionContext;
    }

    @Override
    protected void writeCiProperty(final Object value, final PropertyDescriptor propertyDescriptor, final CiWriter writer) {
        if (value instanceof BaseConfiguration) {
            BaseConfiguration ci = (BaseConfiguration) value;
            Option<PortableConfigurationReference> portableConfigurationReference = PortableConfigurationReference.from(ci);
            if (portableConfigurationReference.nonEmpty()) {
                writer.ciReference(portableConfigurationReference.get().portableReferenceId());
                return;
            }
        }
        super.writeCiProperty(value, propertyDescriptor, writer);
    }

    @Override
    protected void readStringProperty(ConfigurationItem configurationItem, PropertyDescriptor propertyDescriptor, CiReader reader) {
        if (propertyDescriptor.isPassword()) {
            try {
                super.readStringProperty(configurationItem, propertyDescriptor, reader);
            } catch (Exception exception) {
                propertyDescriptor.set(configurationItem, DECRYPTION_FAILED_VALUE);
                logger.info("Failed to read password field: '{}' with value: '{}'", propertyDescriptor, reader.getStringValue(), exception);
            }
        } else {
            super.readStringProperty(configurationItem, propertyDescriptor, reader);
        }
    }

    @Override
    public void resolveReferences(final Repository repository) {
        resolveReferencesWithFallback(repository);
    }

    private static final Logger logger = LoggerFactory.getLogger(UnresolvedReferencesConfigurationItemConverter.class);

    @Override
    public ConfigurationRepository getConfigurationRepository() {
        return configurationRepository;
    }

    @Override
    public ResolutionContext getResolutionContext() {
        return resolutionContext;
    }
}