package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.servlet.*;
import javax.servlet.http.HttpServletResponse;

import com.xebialabs.xlrelease.XLReleaseBootstrapper;

import static java.lang.String.format;
import static javax.servlet.http.HttpServletResponse.SC_GONE;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;

public class VersionEnforcerFilter implements Filter {
    private static final String VERSION_PARAMETER = "xlrv";

    private static final Pattern VERSION_PATTERN = Pattern.compile("^[0-9]+\\.[0-9]+(\\.[0-9]+)?(-(alpha|beta|rc|SNAPSHOT|[0-9]+)(\\.[0-9]+)?)?$");

    private String serverVersion;

    @Override
    public void init(FilterConfig filterConfig) throws ServletException {
        serverVersion = CurrentVersion.get();
    }

    @Override
    public void doFilter(ServletRequest request, ServletResponse response, FilterChain chain) throws IOException, ServletException {
        String[] values = request.getParameterValues(VERSION_PARAMETER);
        boolean doFilter = true;
        if (values != null) {
            String clientVersion = values[0];

            if (!isClientVersionValid(clientVersion)) {
                response.setContentType(TEXT_PLAIN);
                response.getWriter().write(format("%s : provided version format is incorrect", XLReleaseBootstrapper.PRODUCT_DIGITALAI()));
                ((HttpServletResponse) response).setStatus(SC_GONE);
                doFilter = false;
            }

            if (!clientVersion.equals(serverVersion) && doFilter) {
                // Version has changed
                response.setContentType(TEXT_PLAIN);
                response.getWriter().write(format("%s version changed : you are using %s, but server version is now %s", XLReleaseBootstrapper.PRODUCT_DIGITALAI(), clientVersion, serverVersion));
                ((HttpServletResponse) response).setStatus(SC_GONE);
                doFilter = false;
            }
        }

        if (doFilter) {
            chain.doFilter(request, response);
        }
    }

    @Override
    public void destroy() {
    }

    private boolean isClientVersionValid(final String clientVersion) {
        Matcher versionMatcher = VERSION_PATTERN.matcher(clientVersion);
        return versionMatcher.matches();
    }
}
