package com.xebialabs.xlrelease.actors

import akka.actor.{Actor, Props}
import com.xebialabs.xlrelease.actors.UserLastActiveActor.{FlushTokenLastUsed, FlushUsersActive, UpdateLastActive, UpdateTokenLastUsed}
import com.xebialabs.xlrelease.repository.{UserProfileRepository, UserTokenRepository}
import com.xebialabs.xlrelease.utils.FixedSizeMap
import grizzled.slf4j.Logging

import java.util.Date
import java.util.concurrent.Executors
import scala.concurrent.ExecutionContext
import scala.concurrent.duration.FiniteDuration

object UserLastActiveActor {
  def props(userProfileRepository: UserProfileRepository,
            userTokenRepository: UserTokenRepository,
            maxBufferSize: Int, batchSize: Int,
            flushInterval: FiniteDuration): Props =
    Props(new UserLastActiveActor(userProfileRepository, userTokenRepository, maxBufferSize, batchSize, flushInterval))
      .withDispatcher("xl.dispatchers.release-dispatcher")

  sealed trait UpdateLastActiveMessages

  case class UpdateLastActive(canonicalId: String) extends UpdateLastActiveMessages

  case class UpdateTokenLastUsed(tokenId: Integer) extends UpdateLastActiveMessages

  case object FlushUsersActive extends UpdateLastActiveMessages

  case object FlushTokenLastUsed extends UpdateLastActiveMessages
}

/** UserLastActiveActor
  *
  * This actor is used by [[com.xebialabs.xlrelease.service.UserLastActiveActorService]] to accumulate (userId, lastActive) and (tokenId, lastUsed) updates.
  * It keeps updates in a [[FixedSizeMap]] having a capacity of [[maxBufferSize]] userIds.
  * If a userId is already in the buffer, the lastActive is updated to the current timestamp.
  * If a userId is not yet in the buffer and there are less [[maxBufferSize]] userIds, it gets added. If there is not enough room
  * for the new userId, the buffer is flushed to the database and then the user is added.
  *
  * Regardless of how full the buffer is, every [[flushInterval]], all stored updates are flushed to the database.
  *
  * Same as userId, tokenId update and flush will take place.
  *
  * The flushing is done in batches of [[batchSize]], using the provided [[UserProfileRepository]] and [[UserTokenRepository]]repository.
  *
  * @param userProfileRepository
  * @param userTokenRepository
  * @param maxBufferSize
  * @param batchSize
  * @param flushInterval
  */
class UserLastActiveActor(userProfileRepository: UserProfileRepository,
                          userTokenRepository: UserTokenRepository,
                          maxBufferSize: Int, batchSize: Int,
                          flushInterval: FiniteDuration) extends Actor with Logging {

  import context._

  // let's show restraint in threads requirements: just one is enough
  private implicit val ec: ExecutionContext =
    ExecutionContext.fromExecutor(Executors.newSingleThreadExecutor())

  type UserActive = (String, Date)
  type UsersActiveMap = FixedSizeMap[String, Date]

  type TokenUsed = (Integer, Date)
  type TokenUsedMap = FixedSizeMap[Integer, Date]

  private lazy val usersActive: UsersActiveMap = FixedSizeMap.empty(maxBufferSize)
  private lazy val tokensUsed: TokenUsedMap = FixedSizeMap.empty(maxBufferSize)

  override def preStart(): Unit = {
    super.preStart()
    logger.debug(s"Setting up scheduled lastActive and lastUsed flush every $flushInterval")
    system.scheduler.scheduleAtFixedRate(flushInterval, flushInterval, self, FlushUsersActive)
    system.scheduler.scheduleAtFixedRate(flushInterval, flushInterval, self, FlushTokenLastUsed)
  }

  override def postStop(): Unit = {
    flushUsersActive()
    flushTokensUsed()
    super.postStop()
  }

  def receive: Receive = {
    case msg@UpdateLastActive(userId) =>
      implicit val lastActive: Date = now()
      if (!updateOrAddUsersActive(userId)) {
        flushUsersActive()
        self ! msg
      }

    case msg@UpdateTokenLastUsed(tokenId) =>
      implicit val lastUsed: Date = now()
      if (!updateOrAddTokenUsed(tokenId)) {
        flushTokensUsed()
        self ! msg
      }

    case FlushUsersActive =>
      flushUsersActive()

    case FlushTokenLastUsed =>
      flushTokensUsed()
  }

  protected def now(): Date = new Date()

  // in-memory operations:
  protected def updateOrAddUsersActive(userId: String)(implicit lastActive: Date): Boolean = {
    logger.debug(s"updateLastActive($userId)")
    updateUsersActive(userId) || addUsersActive(userId)
  }

  protected def updateUsersActive(userId: String)(implicit now: Date): Boolean = {
    if (usersActive.update(userId -> now)) {
      logger.trace(s"update $userId: $now")
      true
    } else {
      logger.trace(s"update $userId: not found")
      false
    }
  }

  protected def addUsersActive(userId: String)(implicit now: Date): Boolean = {
    if (usersActive.add(userId -> now)) {
      logger.trace(s"add $userId: $now")
      true
    } else {
      logger.trace(s"add $userId: full")
      false
    }
  }

  protected def updateOrAddTokenUsed(tokenId: Integer)(implicit lastUsed: Date): Boolean = {
    logger.debug(s"updateOrAddTokenUsed")
    updateTokensUsed(tokenId) || addTokensUsed(tokenId)
  }

  protected def updateTokensUsed(tokenId: Integer)(implicit now: Date): Boolean = {
    if (tokensUsed.update(tokenId -> now)) {
      true
    } else {
      false
    }
  }

  protected def addTokensUsed(tokenId: Integer)(implicit now: Date): Boolean = {
    if (tokensUsed.add(tokenId -> now)) {
      true
    } else {
      false
    }
  }


  // memory -> database operations:
  protected def flushUsersActive(): Unit = {
    if (usersActive.nonEmpty) {
      logger.debug(s"flushing ${usersActive.size} buffered users lastActive timestamps")
      usersActive.entries
        .grouped(batchSize)
        .foreach(flushUsersActiveBatch)

      usersActive.clear()
    }
  }

  protected def flushUsersActiveBatch(entries: Map[String, Date]): Unit = {
    if (entries.nonEmpty) {
      logger.trace(s"flushBatch(${entries.size})")
      if (entries.size == 1) {
        userProfileRepository.updateLastActive(entries.head._1, entries.head._2)
      } else {
        userProfileRepository.updateLastActiveBatch(entries)
      }
    }
  }

  protected def flushTokensUsed(): Unit = {
    if (tokensUsed.nonEmpty) {
      logger.debug(s"flushing ${tokensUsed.size} buffered token last used timestamps")
      tokensUsed.entries
        .grouped(batchSize)
        .foreach(flushTokensUsedBatch)

      tokensUsed.clear()
    }
  }

  protected def flushTokensUsedBatch(entries: Map[Integer, Date]): Unit = {
    if (entries.nonEmpty) {
      logger.trace(s"flushBatch(${entries.size})")
      if (entries.size == 1) {
        userTokenRepository.updateLastUsed(entries.head._1, entries.head._2)
      } else {
        userTokenRepository.updateLastUsedBatch(entries)
      }
    }
  }

}
