package com.xebialabs.xlrelease.actors.utils

import akka.actor.{ActorSystem, PoisonPill}
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.xlplatform.cluster.ClusterMode.Full
import com.xebialabs.xlrelease.actors._
import com.xebialabs.xlrelease.actors.utils.ReleaseActorTerminator.{ReleaseActorStopped, StopReleaseActor}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.support.akka.spring.{ScalaSpringAwareBean, SpringExtension}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.concurrent.Await
import scala.concurrent.duration._
import scala.language.postfixOps

@Component
class ReleaseActorLifecycleUtils @Autowired()(val actorSystemHolder: ActorSystemHolder,
                                              val xlrConfig: XlrConfig,
                                              val releaseActorService: ReleaseActorService)
  extends ActorLifecycleUtils with ScalaSpringAwareBean {

  private lazy implicit val system: ActorSystem = actorSystemHolder.getInstance()

  def activateReleaseActorAndAwait(releaseId: String) = {
    releaseActorService.activate(releaseId)
    awaitReleaseActorStart(releaseId)
  }

  def terminateAllReleaseActorsAndAwait(terminationTimeout: FiniteDuration) = terminateAndAwait("Release*", terminationTimeout)

  def terminateReleaseActorAndAwait(releaseId: String, terminationTimeout: FiniteDuration) =
    terminateAndAwait(releaseId2ActorName(releaseId), terminationTimeout = terminationTimeout)

  private def terminateAndAwait(absoluteOrReleasePath: String, terminationTimeout: FiniteDuration) = {
    Await.result(findAndTerminate(toActorPath(absoluteOrReleasePath), terminationTimeout = terminationTimeout), 15 seconds)
  }

  private def awaitActorsStart(absoluteOrReleasePath: String) = {
    Await.result(actorStart(toActorPath(absoluteOrReleasePath)), 15 seconds)
  }

  def terminateReleaseActor(releaseId: String): Unit = {
    val actorName = releaseId2ActorName(releaseId)
    val terminator = SpringExtension.actorOf(classOf[ReleaseActorTerminator], s"release-terminator-${actorName}")
    implicit val timeout = Timeout(5.seconds)
    try {
      val stopped = Await.result(terminator ? StopReleaseActor(releaseId), 6.seconds).asInstanceOf[ReleaseActorStopped]
    } finally {
      terminator ! PoisonPill
    }
  }

  private def awaitReleaseActorStart(releaseId: String) =
    awaitActorsStart(releaseId2ActorName(releaseId))

  private def toActorPath(path: String): String =
    if (path.startsWith("/")) {
      path
    } else if (xlrConfig.clusterMode == Full) {
      "/system/sharding/release/*/" + path
    } else {
      "/user/releases/" + path
    }
}
