package com.xebialabs.xlrelease.service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Stream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.codahale.metrics.annotation.Timed;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.xlrelease.api.v1.views.TaskAccessView;
import com.xebialabs.xlrelease.configuration.TaskAccess;
import com.xebialabs.xlrelease.customscripts.ScriptTypes;
import com.xebialabs.xlrelease.domain.ContainerTaskDefinition;
import com.xebialabs.xlrelease.domain.PythonScriptDefinition;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.TaskDefinition;
import com.xebialabs.xlrelease.domain.utils.TaskTypes;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;

import static com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName;
import static com.xebialabs.deployit.security.Permissions.getAuthentication;
import static com.xebialabs.xlrelease.customscripts.ScriptTypes.UNKNOWN_TYPES;
import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

@Service
public class TaskAccessService {

    private ScriptTypes scriptTypes;
    private ConfigurationRepository configurationRepository;
    private ConfigurationService configurationService;
    private RoleService roleService;
    private PermissionEnforcer permissionEnforcer;

    @Autowired
    public TaskAccessService(ScriptTypes scriptTypes, ConfigurationRepository configurationRepository, ConfigurationService configurationService,
                             RoleService roleService, PermissionEnforcer permissionEnforcer) {
        this.scriptTypes = scriptTypes;
        this.configurationRepository = configurationRepository;
        this.configurationService = configurationService;
        this.roleService = roleService;
        this.permissionEnforcer = permissionEnforcer;
    }

    @Timed
    public List<TaskAccessView> getTaskAccesses() {
        List<TaskAccessView> taskAccessViews = getAllTaskAccessViews();

        List<TaskAccess> taskAccesses = readTaskAccesses();

        return mergeWithTaskAccesses(taskAccessViews, taskAccesses);
    }

    private List<TaskAccessView> getAllTaskAccessViews() {
        return Stream.of(TaskTypes.getDefaultTaskTypes(), scriptTypes.getPythonScriptTypes(), scriptTypes.getContainerTaskTypes())
                .flatMap(Collection::stream)
                .map(this::taskToAccessView)
                .collect(toList());
    }

    private TaskAccessView taskToAccessView(Type defaultType) {
        return new TaskAccessView(TaskDefinition.getDisplayGroup(defaultType), defaultType.toString(), TaskDefinition.getDisplayName(defaultType), true, new ArrayList<>());
    }

    private List<TaskAccess> readTaskAccesses() {
        return configurationRepository.findAllByType(Type.valueOf(TaskAccess.class));
    }

    private List<TaskAccessView> mergeWithTaskAccesses(final List<TaskAccessView> taskAccessViews, final List<TaskAccess> taskAccesses) {
        Map<String, TaskAccessView> accessesByTaskType = Maps.uniqueIndex(taskAccessViews, TaskAccessView::getTaskType);
        for (TaskAccess taskAccess : taskAccesses) {
            String taskType = taskAccess.getName();

            if (accessesByTaskType.containsKey(taskType)) {
                TaskAccessView taskAccessView = accessesByTaskType.get(taskType);
                taskAccessView.setAllowedToAll(taskAccess.isAllowedToAll());
                taskAccessView.getRoles().addAll(taskAccess.getRoles());
            }
        }
        return taskAccessViews;
    }

    @Timed
    public List<Type> getAllowedTaskTypesForAuthenticatedUser() {
        List<Role> roles = roleService.getRolesFor(getAuthentication());
        boolean isAdmin = permissionEnforcer.isCurrentUserAdmin();
        Set<String> roleNamesForAuthenticatedUser = new HashSet<>(transform(roles, Role::getName));

        return getTaskAccesses().stream()
                .filter(taskAccess -> isAdmin || taskAccess.isGrantedForRoles(roleNamesForAuthenticatedUser))
                .map(TaskAccessView::toType)
                .collect(toList());
    }

    private <T, R> List<R> transform(final List<T> collection, final Function<T, R> transform) {
        return collection.stream().map(transform).collect(toList());
    }

    @Timed
    public List<TaskDefinition> getTaskDefinitions() {
        final List<Type> allowedTaskTypesForAuthenticatedUser = getAllowedTaskTypesForAuthenticatedUser();

        return transform(getAllTaskAccessViews(), taskAccessView -> {
            final Type type = taskAccessView.toType();

            boolean isAllowed = allowedTaskTypesForAuthenticatedUser.stream().anyMatch(allowedType -> allowedType.equals(type));

            if (PythonScriptDefinition.isScriptDefinition(type)) {
                return new PythonScriptDefinition(type, isAllowed);
            } else if (ContainerTaskDefinition.isContainerTaskDefinition(type)) {
                return new ContainerTaskDefinition(type, isAllowed);
            }
            return new TaskDefinition(type, isAllowed);
        });
    }

    @Timed
    public void checkIfAuthenticatedUserCanUseTask(Task task) {
        checkIfAuthenticatedUserCanUseTasks(Collections.singletonList(task));
    }


    @Timed
    public void checkIfAuthenticatedUserCanUseTasks(List<Task> tasks) {
        checkIfAuthenticatedUserCanUseTaskTypes(transform(tasks, Task::getTaskType));
    }

    @Timed
    public void checkIfAuthenticatedUserCanUseTaskType(Type taskType) {
        checkIfAuthenticatedUserCanUseTaskTypes(Collections.singletonList(taskType));
    }

    @Timed
    public void checkIfAuthenticatedUserCanUseTaskTypes(List<Type> taskTypes) {
        if (null == getAuthenticatedUserName()) {
            return;
        }

        Set<Type> rejectedTaskTypes = new HashSet<>(taskTypes);
        rejectedTaskTypes.removeAll(getAllowedTaskTypesForAuthenticatedUser());

        if (!rejectedTaskTypes.isEmpty() && !rejectedTaskTypes.equals(UNKNOWN_TYPES)) {
            throw PermissionDeniedException.withMessage(format("Task use is not granted to you for types : '%s'", rejectedTaskTypes.toString()));
        }
    }

    @Timed
    public void updateTaskAccesses(List<TaskAccessView> taskAccessesView) {
        taskAccessesView.stream()
                .map(taskAccessView ->
                        new TaskAccess(taskAccessView.getTaskType(), taskAccessView.isAllowedToAll(), taskAccessView.getRoles()))
                .collect(toList())
                .forEach(taskAccess -> configurationService.createOrUpdate(taskAccess));
    }
}
