package com.xebialabs.xlrelease.template

import com.google.common.io.ByteStreams.toByteArray
import com.xebialabs.deployit.repository.{WorkDirContext, WorkDirFactory}
import com.xebialabs.xlrelease.export.TemplateImporter
import grizzled.slf4j.Logging
import org.springframework.core.io.ClassPathResource

import java.io.{ByteArrayInputStream, IOException}
import java.util
import scala.jdk.CollectionConverters._

trait SamplesImporter extends Logging {

  def workDirContext: WorkDirContext

  def templateImporter: TemplateImporter

  protected def importSampleTemplates(sampleTemplates: util.Map[String, String]): Unit = {
    val previousWorkDir = Option(WorkDirContext.get())
    WorkDirContext.initWorkdir(WorkDirFactory.IMPORT_WORKDIR_PREFIX)
    try {
      for ((resource, id) <- sampleTemplates.asScala) {
        importTemplate(resource, id)
      }
    } finally {
      Option(WorkDirContext.get).foreach(_.delete)
      WorkDirContext.clear()
      previousWorkDir.foreach(WorkDirContext.setWorkDir)
    }
  }

  private def importTemplate(classpathResource: String, destinationId: String): Unit = {
    // Load template
    getTemplate(classpathResource) match {
      case Some(template) =>
        // Import into repository
        try {
          val result = templateImporter.importDefaultTemplate(new ByteArrayInputStream(template)).get(0)
          logger.info(s"Imported template $destinationId from $classpathResource")
          for (warning <- result.getWarnings.asScala) {
            logger.info(s"  Warning: $warning")
          }
        } catch {
          case e: Exception =>
            logger.warn(s"Can't import template $destinationId from $classpathResource: ${e.getMessage}")
        }
      case _ =>
    }

  }

  private def getTemplate(classpathResource: String): Option[Array[Byte]] = {
    val in = new ClassPathResource(classpathResource).getInputStream
    try {
      if (in == null) {
        logger.warn(s"Could not find template $classpathResource")
        None
      }
      Some(toByteArray(in))
    } catch {
      case e: IOException =>
        logger.warn(s"Could not load template $classpathResource: ${e.toString}")
        None
    } finally if (in != null) in.close()
  }

}
