package com.xebialabs.xlrelease.db.sql.archiving

import com.xebialabs.deployit.security.Permissions.{getAuthenticatedUserName, getAuthentication}
import com.xebialabs.deployit.security.permission.PlatformPermissions
import com.xebialabs.deployit.security.{PermissionEnforcer, Permissions, RoleService}
import com.xebialabs.xlrelease.api.v1.forms.TimeFrame
import com.xebialabs.xlrelease.db.ArchivedReleases.{REPORT_TASKS_TABLE_ALIAS => TASKS_ALIAS, _}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.Archive.{DEPLOYMENT_TASK_REPORTING_RECORD => DEPL, ITSM_TASK_REPORTING_RECORD => ITSM}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TaskRecordTable
import com.xebialabs.xlrelease.repository.sql.persistence.Table
import com.xebialabs.xlrelease.security.XLReleasePermissions

import java.lang.{Iterable => JIterable}
import java.util.{Date, List => JList}
import scala.collection.mutable.ArrayBuffer
import scala.jdk.CollectionConverters._


abstract class BaseSelectArchivedBuilder[T <: BaseSelectArchivedBuilder[T]](val table: String, val alias: String, val columns: String*)
                                                                           (implicit dialect: Dialect) extends SqlBuilder[T]()(dialect) {

  private var joinedTasks: Boolean = false
  private var joinedDeployments: Boolean = false

  select(s"SELECT ${columns.mkString(", ")} FROM $table $alias")

  def withKind(kind: ReleaseKind): T = {
    if (kind != null) {
      equal(s"$alias.$REPORT_RELEASES_KIND", kind.value())
    }
    self
  }

  def withTags(tags: JIterable[String]): T = if (tags != null) withAllTags(tags.asScala.toList) else self

  def withAllTags(tags: JList[String]): T = if (tags != null) withAllTags(tags.asScala.toList) else self

  def withAllTags(tags: List[String]): T = {
    if (tags != null && tags.nonEmpty) {
      val condition = tags.map(_ =>
        s"""
                $alias.$COMMON_RELEASEID_COLUMN IN
                (
                  SELECT tags.$TAGS_RELEASEID_COLUMN
                  FROM $ARCHIVE_TAGS_TABLE_NAME tags
                  WHERE tags.$TAGS_RELEASEID_COLUMN = $alias.$COMMON_RELEASEID_COLUMN
                  AND tags.$TAGS_VALUE_COLUMN = ?
                )
                """).mkString("(", " AND ", ")")
      conditions += Sql(condition, tags.map(_.toLowerCase))
    }
    self
  }

  def withAnyOfTags(tags: JList[String]): T = if (tags != null) withAnyOfTags(tags.asScala.toList) else self

  def withAnyOfTags(tags: List[String]): T = {
    if (tags != null && tags.nonEmpty) {
      val condition =
        s"""$alias.$COMMON_RELEASEID_COLUMN IN (
              SELECT tags.$TAGS_RELEASEID_COLUMN
              FROM $ARCHIVE_TAGS_TABLE_NAME tags
              WHERE tags.$TAGS_VALUE_COLUMN IN (${tags.map(_ => "?").mkString(",")})
            )"""
      conditions += Sql(condition, tags.map(_.toLowerCase))
    }
    self
  }

  private def securityCriteria(userSpecific: Boolean)
                              (implicit permissionEnforcer: PermissionEnforcer, roleService: RoleService): Option[(String, Seq[String])] = {
    if (userSpecific && Permissions.getAuthentication != null &&
      !permissionEnforcer.hasLoggedInUserPermission(PlatformPermissions.ADMIN) &&
      !permissionEnforcer.hasPermission(Permissions.getAuthentication, XLReleasePermissions.AUDIT_ALL)
    ) {
      val owner = getAuthenticatedUserName
      val roleNames = roleService.getRolesFor(getAuthentication).asScala.map(_.getName)

      val params = ArrayBuffer.empty[String]

      var securityCriteria =
        s"""(
         $alias.$COMMON_RELEASEID_COLUMN IN
         (
          SELECT members.$ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN
          FROM $ARCHIVE_MEMBERVIEWERS_TABLE_NAME members
          WHERE members.$ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN = $alias.$COMMON_RELEASEID_COLUMN
          AND members.$ARCHIVE_MEMBERVIEWERS_USERNAME_COLUMN = ?
         )"""
      params ++= Seq(owner)

      if (roleNames.nonEmpty) {
        securityCriteria +=
          s"""OR
         $alias.$COMMON_RELEASEID_COLUMN IN
         (
          SELECT roles.$ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN
          FROM $ARCHIVE_ROLEVIEWERS_TABLE_NAME roles
          WHERE roles.$ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN = $alias.$COMMON_RELEASEID_COLUMN
          AND roles.$ARCHIVE_ROLEVIEWERS_ROLE_COLUMN IN (${roleNames.map(_ => "?").mkString(",")})
         )"""

        params ++= roleNames
      }
      securityCriteria += ")"

      Option((securityCriteria, params.toSeq))
    } else {
      None
    }
  }

  def withSecurity(userSpecific: Boolean)(implicit permissionEnforcer: PermissionEnforcer, roleService: RoleService): T = {
    securityCriteria(userSpecific) match {
      case Some((securityCriteria, params)) => conditions += Sql(securityCriteria, params)
      case None => ()
    }
    self
  }

  def withSecurityOrOwner(userSpecific: Boolean)(implicit permissionEnforcer: PermissionEnforcer, roleService: RoleService): T = {
    securityCriteria(userSpecific) match {
      case Some((securityCriteria, params)) =>
        val releaseOwner = getAuthenticatedUserName
        if (releaseOwner != null && releaseOwner.nonEmpty) {
          val query = s"($alias.$COMMON_RELEASE_OWNER_COLUMN = ? OR $securityCriteria)"
          val queryParams = releaseOwner +: params
          conditions += Sql(query, queryParams)
        } else {
          conditions += Sql(securityCriteria, params)
        }
      case None => ()
    }
    self
  }

  def withCondition(condition: String, parameters: java.util.Collection[AnyRef]): T = {
    conditions += Sql(condition, parameters.asScala)
    self
  }


  def withStartDate(startDate: Date): T = {
    if (startDate != null) {
      conditions += Sql(s"? <= $alias.$COMMON_END_DATE_COLUMN", Seq(startDate))
    }
    self
  }

  def withEndDate(endDate: Date): T = {
    if (endDate != null) {
      conditions += Sql(s"? > $alias.$COMMON_START_DATE_COLUMN", Seq(endDate))
    }
    self
  }

  def withFolder(folderId: String): T = {
    if (folderId != null) {
      conditions += Sql(s"$alias.$COMMON_RELEASEID_COLUMN LIKE ? ", Seq(Ids.getName(folderId) + "/%"))
    }
    self
  }

  def withOneOfStatuses(statuses: String*): T = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"(${statuses.map(_ => s"$alias.$COMMON_STATUS_COLUMN = ?").mkString(" OR ")})", statuses)
    }
    self
  }

  def withNotNull(column: String): T = {
    conditions += Sql(s"$alias.$column IS NOT NULL", Seq())
    self
  }

  def innerJoin(table: String, dataAlias: String, column1: String, dataColumn2: String): T = {
    addJoin(s"INNER JOIN $table $dataAlias ON $alias.$column1 = $dataAlias.$dataColumn2")
    self
  }

  def withDates(timeFrame: TimeFrame, from: Date, to: Date): T = {
    val startDate = new Date(timeFrame.getStartDate(from))
    val endDate = new Date(timeFrame.getEndDate(to))
    if (startDate != null && endDate != null) {
      conditions += Sql(s"(? <= $alias.$COMMON_END_DATE_COLUMN AND ? > $alias.$COMMON_START_DATE_COLUMN)", Seq(startDate, endDate))
    } else {
      withStartDate(startDate)
      withEndDate(endDate)
    }
    self
  }

  def withPreArchived(preArchived: Boolean = true): T = {
    conditions += Sql(s"$alias.$REPORT_RELEASES_PRE_ARCHIVED = ?", Seq(Integer.valueOf(if (preArchived) 1 else 0)))
    self
  }


  private def selectReleasesHavingTaskFacet(facetTable: Table with TaskRecordTable, facetColumn: String, recordAlias: String, values: Set[String]) = {
    val parameters = values.toSeq.map(_.toLowerCase)
    Sql(
      s"""|SELECT DISTINCT ${recordAlias}Rel.${REPORT_RELEASES_ID_COLUMN} FROM ${REPORT_RELEASES_TABLE_NAME} ${recordAlias}Rel
          |INNER JOIN ${REPORT_TASKS_TABLE_NAME} ${recordAlias}Task ON ${recordAlias}Task.${REPORT_TASKS_RELEASEID_COLUMN} = ${recordAlias}Rel.${REPORT_RELEASES_ID_COLUMN}
          |INNER JOIN ${facetTable.TABLE} $recordAlias ON $recordAlias.${facetTable.TASK_UID} = ${recordAlias}Task.$REPORT_TASKS_UID_COLUMN
          |WHERE LOWER($recordAlias.$facetColumn) IN ${parameters.map(_ => "?").mkString("(", ", ", ")")}""".stripMargin,
      parameters
    )
  }

  protected def whereReleaseHasTaskFacet(table: Table with TaskRecordTable, column: String, recordAlias: String, values: Set[String]): T = {
    val parameters = values.filterNot(_.isEmpty)
    if (parameters.nonEmpty) {
      val Sql(subQuery, subParams) = selectReleasesHavingTaskFacet(table, column, recordAlias, parameters)
      conditions += Sql(s"$alias.$REPORT_RELEASES_ID_COLUMN IN ($subQuery)", subParams)
    }
    self
  }

  def withChangeNumbers(changeNumbers: Set[String]): T = whereReleaseHasTaskFacet(ITSM, ITSM.RECORD, "itsm", changeNumbers)

  def withApplicationNames(applicationNames: Set[String]): T = whereReleaseHasTaskFacet(DEPL, DEPL.APPLICATION_NAME, "app", applicationNames)

  def withEnvironmentNames(environmentNames: Set[String]): T = whereReleaseHasTaskFacet(DEPL, DEPL.ENVIRONMENT_NAME, "env", environmentNames)

  private def joinTasks(): Unit = {
    if (!joinedTasks) {
      addJoin(s"INNER JOIN $REPORT_TASKS_TABLE_NAME ${TASKS_ALIAS} ON ${TASKS_ALIAS}.${REPORT_TASKS_RELEASEID_COLUMN} = $alias.${REPORT_RELEASES_ID_COLUMN}")
      joinedTasks = true
    }
  }

  private def joinDeployments(): Unit = {
    if (!joinedDeployments) {
      joinTasks()
      addJoin(s"INNER JOIN ${DEPL.TABLE} depl ON depl.${DEPL.TASK_UID} = ${TASKS_ALIAS}.${REPORT_TASKS_UID_COLUMN}")
      joinedDeployments = true
    }
  }

  private def self = this.asInstanceOf[T]

}
