package com.xebialabs.xlrelease.api.internal;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Dependency;
import com.xebialabs.xlrelease.domain.GateCondition;
import com.xebialabs.xlrelease.domain.GateTask;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.TaskGranularPermissions;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.views.*;
import com.xebialabs.xlrelease.views.converters.DependencyViewConverter;

import static com.google.common.base.Objects.equal;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE_TASK;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static java.util.stream.Collectors.toList;

/**
 * Gates are special type of task that contains conditions to be fulfilled before the release can continue.
 */
@Path("/gates")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class GatesResource {

    private ReleaseActorService releaseActorService;

    private GateConditionService conditions;

    private ReleaseService releaseService;

    private PermissionChecker permissions;

    private DependencyService dependencyService;

    private TaskService taskService;

    private TaskGranularPermissions taskPermissionChecker;

    @Autowired
    public GatesResource(ReleaseActorService releaseActorService,
                         GateConditionService conditions,
                         ReleaseService releaseService,
                         PermissionChecker permissions,
                         DependencyService dependencyService,
                         TaskService taskService,
                         TaskGranularPermissions taskPermissionChecker) {
        this.releaseActorService = releaseActorService;
        this.conditions = conditions;
        this.releaseService = releaseService;
        this.permissions = permissions;
        this.dependencyService = dependencyService;
        this.taskService = taskService;
        this.taskPermissionChecker = taskPermissionChecker;
    }

    @PUT
    @Timed
    @Path("conditions/{conditionId:.*Condition[^/-]*}")
    public GateConditionView updateCondition(@PathParam("conditionId") @IdParam String conditionId, GateConditionView conditionView) {
        GateConditionView gateConditionView;

        final String releaseId = releaseIdFrom(conditionId);
        final boolean isTemplate = releaseService.isTemplate(releaseId);

        // TODO make this a single permission check!
        if (isTemplate && permissions.hasPermission(EDIT_TEMPLATE, releaseId)) {
            gateConditionView = new GateConditionView(releaseActorService.updateGateCondition(conditionId, conditionView.toCondition()));

        } else if ((permissions.isAllowedToWorkOnTask(getParentId(conditionId)) &&
                checkHasEditTaskPermissionIfConditionTitleChanged(conditionId, conditionView, releaseId)) ||
                taskPermissionChecker.hasEditTaskConfigurationGranularPermission(releaseId)) {

            gateConditionView = new GateConditionView(releaseActorService.updateGateCondition(conditionId, conditionView.toCondition()));
        } else {
            throw PermissionDeniedException.forPermission(isTemplate ? EDIT_TEMPLATE : EDIT_RELEASE_TASK, releaseId);
        }

        return gateConditionView;
    }

    @POST
    @Timed
    @Path("{gateId:.*Task[^/-]*}/conditions")
    public GateConditionView createCondition(@PathParam("gateId") @IdParam String gateId) {
        taskPermissionChecker.checkHasEditTaskOrEditTaskConfiguration(releaseIdFrom(gateId));

        GateCondition gateCondition = releaseActorService.createGateCondition(gateId);
        return new GateConditionView(gateCondition);
    }

    @DELETE
    @Timed
    @Path("conditions/{conditionId:.*Condition[^/-]*}")
    public void deleteCondition(@PathParam("conditionId") @IdParam String conditionId) {
        taskPermissionChecker.checkHasEditTaskOrEditTaskConfiguration(releaseIdFrom(conditionId));
        releaseActorService.deleteGateCondition(conditionId);
    }

    // TODO: try to merge this method with getDependenciesStatuses later
    @GET
    @Timed
    @Path("{gateId:.*Task[^/-]*}/dependencies")
    public List<DependencyView> getDependencies(@PathParam("gateId") @IdParam String gateId) {
        GateTask gateTask = (GateTask) taskService.findByIdIncludingArchived(gateId);
        permissions.checkViewTask(gateTask);

        return gateTask.getDependencies().stream().map(DependencyViewConverter::toDependencyView).collect(toList());
    }

    @POST
    @Timed
    @Path("{gateId:.*Task[^/-]*}/dependencies")
    public DependencyView addDependency(@PathParam("gateId") @IdParam String gateId, DependencyView dependencyView) {
        taskPermissionChecker.checkHasEditTaskOrEditTaskConfiguration(releaseIdFrom(gateId));
        if (dependencyView.hasFixedTarget()) {
            permissions.checkView(releaseIdFrom(dependencyView.toTargetId()));
        }

        Dependency dependency = releaseActorService.createDependency(gateId, dependencyView.toTargetIdOrVariable());
        return DependencyViewConverter.toDependencyView(dependency);
    }

    @GET
    @Timed
    @Path("{gateId:.*Task[^/-]*}/dependency-target-candidates")
    public List<PlanItemView> getDependencyTargetCandidates(@PathParam("gateId") @IdParam String gateId) {
        List<Release> dependencyTargetCandidates = dependencyService.findAllDependencyCandidates(gateId);
        return dependencyTargetCandidates.stream().map(PlanItemView::new).collect(toList());
    }

    @GET
    @Timed
    @Path("{gateId:.*Task[^/-]*}/dependency-target-candidates/{releaseId:.*Release[^/-]*}")
    public ReleaseView loadDependencyTargetCandidate(@PathParam("gateId") @IdParam String gateId, @PathParam("releaseId") @IdParam String releaseId) {
        permissions.checkView(releaseId);
        return new ReleaseView(dependencyService.getDependencyCandidate(gateId, releaseId));
    }

    @GET
    @Timed
    @Path("{gateId:.*Task[^/-]*}/dependency-target-statuses")
    public List<DependencyTargetView> getDependenciesStatuses(@PathParam("gateId") @IdParam String gateId) {
        GateTask gateTask = (GateTask) taskService.findByIdIncludingArchived(gateId, ResolveOptions.WITH_DECORATORS());
        permissions.checkViewTask(gateTask);
        return gateTask.getDependencies().stream()
                .map(dependency -> {
                    String targetId = dependency.getTargetId();
                    try {
                        IdAndStatus r = dependencyService.findDependencyTargetByTargetId(targetId);
                        return new DependencyTargetView(r.id(), r.status());
                    } catch (NotFoundException ex) {
                        return new DependencyTargetView(targetId, "not_found");
                    }
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private boolean checkHasEditTaskPermissionIfConditionTitleChanged(final String conditionId, final GateConditionView conditionView, final String releaseId) {
        if (!equal(conditions.findById(conditionId).getTitle(), conditionView.toCondition().getTitle())) {
            return permissions.hasPermission(EDIT_RELEASE_TASK, releaseId);
        } else {
            return true;
        }
    }

}
