package com.xebialabs.xlrelease.actors.utils

import akka.actor.{Actor, ActorIdentity, ActorRef, Identify, Terminated}
import akka.event.slf4j.SLF4JLogging
import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor.GracefulPoisonPill
import com.xebialabs.xlrelease.actors.ReleasesActorHolder
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.actors.utils.ReleaseActorTerminator.{ReleaseActorStopped, StopReleaseActor}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.support.akka.spring.SpringActor

import java.util.UUID

@SpringActor
class ReleaseActorTerminator(releaseActorHolder: ReleasesActorHolder) extends Actor with SLF4JLogging {
  override def receive: Receive = {
    case StopReleaseActor(releaseId) =>
      val requestId = UUID.randomUUID().toString
      val releaseRootActor = releaseActorHolder.awaitActorRef()
      val realReleaseId = releaseIdFrom(releaseId)
      releaseRootActor ! ReleaseAction(realReleaseId, Identify(requestId))
      context.become(stopRequested(realReleaseId, requestId, sender()))
  }

  private def stopRequested(releaseId: String, requestId: String, requester: ActorRef): Receive = {
    case ActorIdentity(correlationId, maybeRef) if correlationId == requestId =>
      maybeRef match {
        case Some(releaseRootActor) =>
          log.info(s"Terminating ${releaseRootActor.path}")
          context.watch(releaseRootActor)
          releaseRootActor ! ReleaseAction(releaseIdFrom(releaseId), GracefulPoisonPill)
          context.become(watching(releaseId, releaseRootActor, requester))
        case None =>
          log.error("no release supervisor found. terminating")
          context.stop(self)
      }
    case msg =>
      log.error("stopRequested: Unexpected message received {}", msg)
      context.stop(self)
  }

  private def watching(releaseId: String, releaseSupervisorRef: ActorRef, requester: ActorRef): Receive = {
    case Terminated(terminatedRef) if releaseSupervisorRef == terminatedRef =>
      requester ! ReleaseActorStopped(releaseId, releaseSupervisorRef)
      context.stop(self)
    case msg =>
      log.error("watching: Unexpected message received {}", msg)
      context.stop(self)
  }
}


object ReleaseActorTerminator {
  sealed trait Msg
  case class StopReleaseActor(releaseId: String) extends Msg

  case class ReleaseActorStopped(releaseId: String, actorRef: ActorRef) extends Msg

}