package com.xebialabs.xlrelease.scheduler

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.runner.domain.{JobId, RunnerId}
import com.xebialabs.xlrelease.scheduler.converters.TaskJobConverter
import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner
import com.xebialabs.xlrelease.scheduler.events.{JobCreatedEvent, JobFinishedEvent, JobReservedEvent}
import com.xebialabs.xlrelease.scheduler.repository.{ConfirmJobExecution, DeleteById, JobRepository, ReserveJob}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.context.annotation.{Lazy, Scope, ScopedProxyMode}
import org.springframework.stereotype.Component

import java.util.concurrent.DelayQueue
import java.util.concurrent.atomic.AtomicBoolean
import java.util.function.Predicate
import scala.util.{Failure, Success, Try}


@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
class DefaultJobQueue(val taskJobConverter: TaskJobConverter,
                      val jobRepository: JobRepository,
                      broadcastService: BroadcastService,
                      @Lazy
                      val actorSystemHolder: ActorSystemHolder)
  extends Logging with NodeId {

  private val _localJobs: DelayQueue[Job] = new DelayQueue[Job]()
  private val _isRunning = new AtomicBoolean(true)

  @Timed
  def submit(job: Job): Unit = {
    logger.debug(s"submitted job $job")
    job match {
      case job: TaskJob[_] =>
        job.setId(jobRepository.create(JobRow(job).copy(node = nodeId)).id)
      case _ => () // special case when we send a msg that does not have task id (i.e. StopWorkerThread)
    }
    addToLocalJobs(job, broadcastJob = job.broadcast)
  }

  def submitExisting(job: Job): Unit = {
    logger.debug(s"submitted existing job $job")
    removeJob(job)
    addToLocalJobs(job, broadcastJob = job.broadcast)
  }

  def submitBroadcasted(job: Job): Unit = {
    logger.debug(s"submitted broadcasted job $job")
    removeJob(job)
    addToLocalJobs(job, broadcastJob = false)
  }

  @Timed
  def replace(job: Job): Unit = {
    logger.debug(s"replaced job $job")
    val newJobId = jobRepository.replace(JobRow(job).copy(node = nodeId, runnerId = null)).id
    removeJob(job)
    job.setId(newJobId)
    addToLocalJobs(job, broadcastJob = false)
  }

  def isRunning(): Boolean = {
    _isRunning.get()
  }

  def localJobs: DelayQueue[Job] = {
    _localJobs
  }

  def reserve(job: Job, runnerId: String = LocalJobRunner.getId()): Job = {
    val lockAttempt = jobRepository.update(ReserveJob(JobRow(job), nodeId, runnerId))
    lockAttempt match {
      case Failure(exception) => throw exception
      case Success(row) =>
        val lockedJob = job match {
          case taskJob: TaskJob[_] => taskJobConverter.fromJobRow(row, Some(taskJob.taskRef)) // row -> taskJob
          case _ => taskJobConverter.fromJobRow(row) // row -> taskJob
        }
        broadcastService.broadcast(JobReservedEvent(lockedJob.id), publishEventOnSelf = false)
        // safe to remove job that is not there
        removeJob(job)
        lockedJob
    }
  }

  def confirm(runnerId: RunnerId, jobId: JobId): (Boolean, Option[TaskJob[_]]) = {
    val confirmationResult = for {
      jobRow <- Try(jobRepository.read(jobId).get)
      confirmedRow <- jobRepository.update(ConfirmJobExecution(jobRow, runnerId))
    } yield confirmedRow
    confirmationResult match {
      case Failure(ex) =>
        logger.warn(s"Unable to confirm job $jobId", ex)
        (false, None)
      case Success(row) =>
        val maybeJob = Try(taskJobConverter.fromJobRow(row)) match {
          case Failure(_) => None // if job is not a task job
          case Success(value) => Option(value)
        }
        (true, maybeJob)
    }
  }

  @Timed
  def finish(jobId: JobId): Unit = {
    logger.debug(s"finishing job $jobId")
    removeJob(jobId)
    jobRepository.read(jobId).foreach { job =>
      jobRepository.delete(DeleteById(job.id))
      broadcastService.broadcast(JobFinishedEvent(job.executionId, job.id), publishEventOnSelf = true)
    }
  }

  def start(): Unit = {
    logger.debug(s"starting job queue")
    _isRunning.set(true)
  }

  def stop(): Unit = {
    logger.debug(s"stopping job queue")
    _isRunning.set(false)
  }

  def cancelIf(predicate: Predicate[Job]): Unit = log("cancelIf") {
    localJobs.removeIf(predicate)
  }

  def removeJob(job: Job): Unit = log("removeJob") {
    localJobs.remove(job)
  }

  def removeJob(jobId: JobId): Unit = log("removeJob by id") {
    localJobs.removeIf(_.id == jobId)
  }

  def size(): Int = {
    localJobs.size()
  }

  private def addToLocalJobs(job: Job, broadcastJob: Boolean): Unit = log("addToLocalJobs") {
    if (localJobs.contains(job) && !job.isInstanceOf[StopWorkerThread]) {
      val ex = new IllegalStateException(s"Queue already contains job $job")
      logger.error("Please report an error.", ex)
    }
    localJobs.add(job)
    if (broadcastJob) {
      broadcastService.broadcast(JobCreatedEvent(job), publishEventOnSelf = false)
    }
  }

  private def log(prefix: String)(block: => Unit): Unit = {
    logger.trace(s"$prefix: before ${localJobsSize()}")
    try {
      block
    } finally {
      logger.trace(s"$prefix: after: ${localJobsSize()}")
    }
  }

  private def localJobsSize(): String = {
    import scala.jdk.CollectionConverters._
    s"${size()}: ${localJobs.iterator().asScala.map(_.id).mkString(",")}"
  }
}
