package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.core.xml.WriteWithoutPasswordCiConverter
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.{RepositoryAdapter, WorkDirContext}
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, ReleasesFilters}
import com.xebialabs.xlrelease.builder.ReleaseBuilder.newRelease
import com.xebialabs.xlrelease.db.ArchivedReleases.AttachmentInfo
import com.xebialabs.xlrelease.db.{ArchivedReleases, ArchivedReleasesSearch}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.INACTIVE_STATUSES
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, FullReleaseId}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.{newEncryptingConverter, serialize}
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.joda.time.format.DateTimeFormat
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.retry.support.{RetryTemplate, RetryTemplateBuilder}
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Isolation.READ_COMMITTED
import org.springframework.transaction.annotation.Propagation.REQUIRED
import org.springframework.transaction.annotation.Transactional

import java.util
import java.util.{Date, Locale, List => JList, Set => JSet}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

object ArchivingService extends Logging {

  def getMonthYear(startDate: DateTime): String = {
    s"${DateTimeFormat.forPattern("MM").withLocale(Locale.ENGLISH).print(startDate)}.${startDate.getYear}"
  }
}

/**
 * The service responsible for storing releases into the relational database once they are completed.
 */
//scalastyle:off number.of.methods
@Service
@Transactional(value = "reportingTransactionManager", propagation = REQUIRED, isolation = READ_COMMITTED, rollbackFor = Array(classOf[Throwable]))
class ArchivingService @Autowired()(val archivedReleases: ArchivedReleases,
                                    archivedReleasesSearch: ArchivedReleasesSearch,
                                    archivedActivityLogsSerializer: ArchivedActivityLogsSerializer,
                                    val repositoryAdapter: RepositoryAdapter,
                                    completedReleasesExportService: CompletedReleaseExportService,
                                    releaseRepository: ReleaseRepository,
                                    dependencyRepository: DependencyRepository,
                                    releaseActorService: ReleaseActorService,
                                    teamRepository: TeamRepository,
                                    securedCis: SecuredCis,
                                    releaseExtensionsRepository: ReleaseExtensionsRepository,
                                    commentRepository: CommentRepository,
                                    facetRepositoryDispatcher: FacetRepositoryDispatcher) extends ArchivedReleaseReader with Logging {

  facetRepositoryDispatcher.setupGenericFacetArchiveRepository(this)

  val retryTemplate: RetryTemplate = new RetryTemplateBuilder().fixedBackoff(30000).maxAttempts(5).build()

  @Timed
  def archiveRelease(releaseId: String): Unit = {
    val release = tryReadRelease(releaseId)

    if (release == null) {
      logger.warn(s"Release [$releaseId] cannot be read from active database, proceeding to archive release and remove from active database")
      deleteRelease(releaseId)
      markAsArchived(releaseId)
    } else if (!shouldArchive(release)) {
      logger.debug(s"Removing non archive-able release [$releaseId] from active database")
      // let's assume do-not-archive releases are not going to be connected, if yes, add archiveDependencies(release) here
      deleteRelease(releaseId)
    } else if (!INACTIVE_STATUSES.contains(release.getStatus)) {
      logger.warn(s"Received request to archive release [$releaseId] with invalid status [${release.getStatus}], " +
        s"proceeding to archive release and remove from active database")
      release.abort(s"Aborting release due to invalid status ${release.getStatus} on archival")
      archiveAttachments(release)
      archiveDependencies(release)
      deleteRelease(releaseId)
      markAsArchived(releaseId)
    } else {
      ensurePreArchived(release)
      decorateRelease(release)
      archiveAttachments(release)
      archiveDependencies(release)
      archiveReleaseAndLogs(releaseId, release)
      deleteRelease(releaseId)
      markAsArchived(releaseId)
    }
  }

  @Timed
  def deletePreArchiveRelease(releaseId: String): Boolean = archivedReleases.deleteReleaseFromArchive(releaseId)

  @Timed
  def preArchiveRelease(release: Release): Unit = {
    if (existsPreArchived(release.getId)) {
      logger.debug(s"Release [${release.getId}] is already pre-archived, nothing to do")
    } else {
      decorateRelease(release, withComments = false, withExtensions = false)
      val serializedRelease = serializeRelease(release)

      if (shouldArchive(release)) {
        logger.debug(s"Copying release [${release.getId}] to archive database, marking as pre-archived")

        archivedReleases.insert(release, serializedRelease, "", preArchived = true)
        archiveReportingRecords(release)
        releaseRepository.setPreArchived(release.getId, preArchived = true)
      }
      logger.debug(s"Running release [${release.getId}] through export hooks")
      completedReleasesExportService.sendThroughExportHooks(release, serializedRelease)
    }
  }

  private def serializeRelease(release: Release): String = {
    val releaseJson = Try(serialize(release, new WriteWithoutPasswordCiConverter())) match {
      case Success(serialized) => serialized
      case Failure(e) =>
        logger.warn(s"Unable to serialize $release, continuing without original JSON", e)
        serialize(newRelease().withId(release.getId).withTitle(release.getTitle)
          .withStartDate(release.getStartDate)
          .withEndDate(release.getEndDate)
          .withStatus(release.getStatus)
          .withDescription(s"Non-recoverable exception serializing release: ${e.getMessage}").build())
    }
    releaseJson
  }

  private def serializeActivityLogs(release: Release): String = {
    Try(archivedActivityLogsSerializer.serializeActivityLogsOf(release.getId)) match {
      case Success(activityLogsJson) => activityLogsJson
      case Failure(e) =>
        logger.warn(s"Unable to serialize activity logs of $release, continuing without logs", e)
        ""
    }
  }

  private def tryReadRelease(releaseId: String) = {
    Try(repositoryAdapter.read[Release](releaseId)).recover {
      case e: Throwable => logger.warn(s"Unable to read release [$releaseId] from active database", e)
        null
    }.get
  }

  private def ensurePreArchived(release: Release): Unit = {
    logger.debug(s"Checking release [${release.getId}] is pre-archived")
    try {
      preArchiveRelease(release)
    } catch {
      case e: Exception => logger.warn(s"Unable to pre-archive $release, proceeding to archive release", e)
    }
  }

  private def decorateRelease(release: Release, withComments: Boolean = true, withExtensions: Boolean = true): Unit = {
    logger.debug(s"Decorating release [${release.getId}]")
    try {
      if (withExtensions) releaseExtensionsRepository.decorate(release)
      if (withComments) commentRepository.decorate(release)
      release.setTeams(teamRepository.getTeams(securedCis.getEffectiveSecuredCi(release.getId)))
      val facets = facetRepositoryDispatcher.liveRepository.genericRepository.findAllFacetsByRelease(release)
      release.getAllTasks.asScala.foreach { task =>
        task.setFacets(new util.ArrayList(facets.filter(_.getTargetId == task.getId).asJava))
      }
    } catch {
      case e: Exception => logger.warn(s"Unable to decorate $release, proceeding to archive release as-is", e)
    }
  }

  private def archiveAttachments(release: Release): Unit = {
    logger.debug(s"Archiving attachments for [${release.getId}]")
    val previousWorkDirContext = Option(WorkDirContext.get())
    WorkDirContext.initWorkdir()
    try {
      release.getAttachments.asScala.foreach { attachment =>
        Try(attachment.getFile.getInputStream).foreach { data =>
          try {
            archivedReleases.insertAttachment(release.getId, AttachmentInfo(attachment.getId, attachment.getFile.getName, data))
          } finally {
            data.close()
          }
        }
      }
    } catch {
      case e: Exception => logger.warn(s"Unable to archive attachments of $release", e)
    } finally {
      Option(WorkDirContext.get).foreach(_.delete())
      WorkDirContext.clear()
      previousWorkDirContext.foreach(WorkDirContext.setWorkDir)
    }
  }

  private def archiveDependencies(release: Release): Unit = {
    logger.debug(s"Archiving dependencies for [${release.getId}]")
    try {
      retryTemplate.execute { context =>
        if (context.getRetryCount > 0)
          logger.warn(s"Retrying archival of dependencies for [${release.getId}]. Error: ${context.getLastThrowable}.")
        archiveAllOutgoingDependencies(release)
        archiveAllIncomingDependencies(release.getId)
      }
    } catch {
      case e: Exception => logger.warn(s"Unable to archive dependencies of $release", e)
    }
  }

  private def archiveReleaseAndLogs(releaseId: String, release: Release): Unit = {
    logger.debug(s"Archiving activity logs and release JSON for [$releaseId]")
    val releaseJson = serializeRelease(release)
    val activityLogsJson = serializeActivityLogs(release)
    try {
      if (!archivedReleases.update(releaseId, releaseJson, activityLogsJson)) {
        logger.warn(s"Could not update archived release JSON [$releaseId]")
      }
      archivedReleases.updateViewers(releaseId, release.getTeams.asScala.toSeq)
    } catch {
      case e: Throwable => logger.warn(s"Unable to update $release in archive", e)
    }
  }

  private def archiveReportingRecords(release: Release): Unit = {
    logger.debug(s"Archiving reporting records for [${release.getId}]")
    val specialTypes = facetRepositoryDispatcher.supportedTypes
    facetRepositoryDispatcher.liveRepository.findAllFacetsByRelease(release)
      .filter(facet => specialTypes.exists(facet.getType.instanceOf))
      .foreach { liveFacet =>
        Try(facetRepositoryDispatcher.archiveRepository.create(liveFacet)).recover {
          case e: Throwable => logger.warn(s"Unable to pre-archive reporting record [${liveFacet.getId}]", e)
        }
      }
  }

  private def deleteRelease(releaseId: String): Unit = if (releaseRepository.exists(releaseId)) {
    logger.debug(s"Deleting [$releaseId] from active database")
    releaseRepository.delete(releaseId, failIfReferenced = false)
  }

  private def markAsArchived(releaseId: String): Unit = {
    archivedReleases.setPreArchived(releaseId, preArchived = false)
  }

  def shouldArchive(release: Release): Boolean = {
    !release.isTutorial && release.isArchiveRelease
  }

  def getRelease(releaseId: String): Release =
    getRelease(releaseId, includePreArchived = false)

  @Timed
  def getRelease(releaseId: String, includePreArchived: Boolean): Release = {
    getReleaseOption(releaseId, includePreArchived).getOrElse {
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    }
  }

  @Timed
  def getReleaseTitle(releaseId: String): String = {
    archivedReleases.getReleaseTitle(releaseId).getOrElse {
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    }
  }

  @Timed
  def getReleaseOwner(releaseId: String): String = {
    archivedReleases.getReleaseOwner(releaseId).getOrElse {
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    }
  }

  @Timed
  def getReleaseKind(releaseId: String): ReleaseKind = {
    val kind = archivedReleases.getReleaseKind(releaseId).getOrElse {
        throw new NotFoundException(s"Could not find archived release [$releaseId]")
      }
    ReleaseKind.valueOf(kind.toUpperCase)
  }

  @Timed
  def getPhase(phaseId: String): Phase = {
    getReleaseItem[Phase](phaseId)(_.getPhase).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(phaseId)}]")
    }
  }

  @Timed
  def getTask(taskId: String): Task = {
    getReleaseItem[Task](taskId)(_.getTask).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(taskId)}]")
    }
  }

  @Timed
  def getVariable(variableId: String): Variable = {
    getVariableOption(variableId).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(variableId)}]")
    }
  }

  @Timed
  def exists(ciId: String): Boolean = {
    if (isReleaseId(ciId)) {
      archivedReleases.exists(ciId)
    } else if (isPhaseId(ciId)) {
      getReleaseItem[Phase](ciId)(_.getPhase).isDefined
    } else if (isTaskId(ciId)) {
      getReleaseItem[Task](ciId)(_.getTask).isDefined
    } else if (isVariableId(ciId)) {
      getVariableOption(ciId).isDefined
    } else {
      false
    }
  }

  @Timed
  def existsPreArchived(ciId: String): Boolean = {
    if (isReleaseId(ciId)) {
      archivedReleases.existsPreArchived(ciId)
    } else {
      false
    }
  }

  @Timed
  def existsByName(releaseName: String): Boolean = {
    archivedReleases.existsByName(releaseName)
  }

  @Timed
  def attachmentExists(attachmentId: String): Boolean = {
    archivedReleases.attachmentExists(attachmentId)
  }

  @Timed
  def getAttachment(attachmentId: String): Attachment = {
    getAttachmentOption(attachmentId).getOrElse {
      throw new NotFoundException(s"Could not find archived attachment [$attachmentId]")
    }
  }

  @Timed
  def getAttachmentOption(attachmentId: String): Option[Attachment] = {
    for {
      release <- getReleaseOption(releaseIdFrom(attachmentId))
      attachment <- Option(release.getAttachment(attachmentId))
      overthereFile <- archivedReleases.getAttachment(attachmentId)
    } yield {
      attachment.setFile(overthereFile)
      attachment
    }
  }

  @Timed
  def searchReleases(filters: ReleasesFilters): JList[Release] = {
    searchReleases(filters, None, None)
  }

  @Timed
  def searchReleases(filters: ReleasesFilters, limit: Long, offset: Long): JList[Release] = {
    searchReleases(filters, Option(limit), Option(offset))
  }

  @Timed
  def findShortReleaseIdsWithFolderNameAndOrderCriterion(filters: Seq[ReportFilter],
                                                         order: Option[ReleaseOrderMode],
                                                         limit: Option[Long] = None,
                                                         offset: Option[Long] = None
                                                        ): List[(AdaptiveReleaseId, Any)] = {
    archivedReleasesSearch.searchReleaseIdsAndOrderCriterion(filters, order, limit, offset)
  }

  @Timed
  def countReleasesByStatus(filters: ReleasesFilters): Map[ReleaseStatus, Int] = {
    archivedReleasesSearch.countReleasesByStatus(filters)
  }

  @Timed
  def countReleases(filters: Seq[ReportFilter]): Map[ReleaseStatus, Int] = {
    archivedReleasesSearch.countReleasesByStatus(filters)
  }

  @Timed
  def findArchivableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    archivedReleases.findArchivableReleaseIds(date, pageSize)
  }

  @Timed
  def findPurgableReleaseIds(date: Date, pageSize: Int): Seq[String] = {
    archivedReleases.findPurgableReleaseIds(date, pageSize)
  }

  @Timed
  def purgeArchivedRelease(releaseId: String): Unit = {
    archivedReleases.deleteReleaseFromArchive(releaseId)
  }

  @Timed
  def archiveAllIncomingDependencies(releaseId: String): Unit = {
    val incomingDependencies = dependencyRepository.findAllIncomingDependencies(Seq(releaseId), Seq.empty, referencingChildren = true)
    incomingDependencies
      .groupBy(d => normalizeId(releaseIdFrom(d.getId)))
      .foreach {
        case (referencingReleaseId, rawDependenciesToArchive) =>
          releaseActorService.archiveDependencies(referencingReleaseId, rawDependenciesToArchive.map(_.getId))
      }
  }

  @Timed
  def getAllTags(limitNumber: Int): JSet[String] = {
    archivedReleases.findAllTags(limitNumber).asJava
  }

  @Timed
  def isWorkflow(releaseId: String): Boolean = {
    ReleaseKind.WORKFLOW == getReleaseKind(releaseId)
  }

  private def searchReleases(filters: ReleasesFilters, limit: Option[Long], offset: Option[Long]): JList[Release] = {
    val releaseJsons = archivedReleasesSearch.searchReleases(filters, limit, offset)
    releaseJsons.map(json => deserializeArchivedRelease(json, repositoryAdapter, newEncryptingConverter()))
  }.asJava

  def searchReleasesByReleaseIds(releaseIds: Seq[String]): Seq[Release] = {
    val releaseJsons = archivedReleasesSearch.searchReleasesByIds(releaseIds)
    releaseJsons.map(json => deserializeArchivedRelease(json, repositoryAdapter, newEncryptingConverter()))
  }

  def searchReleasesBasicExtByReleaseIds(releaseIds: Seq[String]): Seq[ReleaseBasicDataExt] =
    archivedReleasesSearch.searchReleasesBasicInfoByIds(releaseIds.map(FullReleaseId(_)
      .withOnlyOneParentOrApplicationsForArchiveDb()))

  private def getReleaseItem[T <: PlanItem](itemId: String)(getItem: Release => String => T): Option[T] =
    getReleaseOption(releaseIdFrom(itemId))
      .flatMap(r => Option(getItem(r)(itemId)))

  private def getVariableOption(variableId: String): Option[Variable] = {
    getReleaseOption(releaseIdFrom(variableId)).flatMap { release =>
      release.getVariables.asScala.find(_.getId == variableId)
    }
  }

  private def archiveAllOutgoingDependencies(release: Release): Unit = {
    val dependencies = release.getAllGates.asScala
      .flatMap(_.getDependencies.asScala)
      .filter(!_.isArchived)
    dependencies.foreach(_.archive())
    // no need for DB operations, release deletion will remove all outgoing references
  }
}
