package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.repository.ItemAlreadyExistsException
import com.xebialabs.xlrelease.api.v1.filter.CategoryFilters
import com.xebialabs.xlrelease.domain.Category
import com.xebialabs.xlrelease.domain.events.{CategoryCreatedEvent, CategoryDeletedEvent, CategoryUpdatedEvent}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.CategoryRepository
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, Pageable}
import org.springframework.util.StringUtils

class CategoryService(categoryRepository: CategoryRepository, eventBus: EventBus) extends Logging {

  @Timed
  def findBy(categoryFilters: CategoryFilters, pageable: Pageable): Page[Category] = {
    categoryRepository.findBy(categoryFilters, pageable)
  }

  @Timed
  def create(category: Category): Category = {
    category.setCiUid(null)
    validateAndFixCategory(category)
    checkForDuplicates(category)
    val createdCategory = categoryRepository.create(category)
    eventBus.publish(CategoryCreatedEvent(createdCategory))
    createdCategory
  }

  @Timed
  def update(category: Category): Category = {
    if (category.getCiUid == null) {
      throw new IllegalArgumentException(s"The category ciUid must not be null or empty")
    }
    categoryRepository.findByCiUid(category.getCiUid) match {
      case Some(originalCategory) =>
        validateAndFixCategory(category)

        // If category is renamed make sure the new name doesn't already exist
        if (originalCategory.getTitle.toLowerCase() != category.getTitle.toLowerCase()) {
          checkForDuplicates(category)
        }

        val updatedCategory = categoryRepository.update(category)
        eventBus.publish(CategoryUpdatedEvent(originalCategory, updatedCategory))
        updatedCategory
      case None =>
        throw new IllegalArgumentException(s"The category '${category.getCiUid}' does not exist")
    }
  }

  @Timed
  def delete(id: Int): Unit = {
    categoryRepository.findByCiUid(id) match {
      case Some(category) =>
        categoryRepository.delete(id)
        eventBus.publish(CategoryDeletedEvent(category))
      case None => ()
    }
  }

  private def validateAndFixCategory(category: Category): Unit = {
    var categoryStr = category.getTitle
    if (!StringUtils.hasText(categoryStr)) {
      throw new IllegalArgumentException(s"The category title must not be null or empty")
    }
    categoryStr = Category.sanitizeTitle(categoryStr)

    category.setTitle(categoryStr)
    if (null == category.getActive) {
      category.setActive(true)
    }
  }

  private def checkForDuplicates(category: Category): Unit = {
    val found = categoryRepository.findByTitle(category.getTitle)

    found match {
      case Some(existingCategory) =>
        // Compare ids so it works with create and update
        if (category.getCiUid != existingCategory.getCiUid) {
          throw new ItemAlreadyExistsException("The category '%s' already exists", category.getTitle)
        }
      case None =>
    }
  }
}
