package com.xebialabs.xlrelease.scheduler

import com.xebialabs.xlrelease.actors.{ActorSystemHolder, ReleaseActorService}
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.service.JobService
import com.xebialabs.xlrelease.user.User
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable

import scala.util.Try

trait JobRecoveryLogic extends Logging with NodeId {
  def jobQueue: JobQueue

  def jobService: JobService

  def releaseActorService: ReleaseActorService

  def actorSystemHolder: ActorSystemHolder

  protected def recoverAllJobs(): Unit = {
    val foundJobs = findAllJobs()
    logger.debug(s"Found ${foundJobs.size} jobs, that should be failed or recovered")
    foundJobs.foreach(failOrRecoverAbandonedJob)
  }

  protected def recoverJobs(nodeId: String): Unit = {
    val foundJobs = findJobsForNode(nodeId)
    logger.debug(s"Found ${foundJobs.size} jobs assigned to node = $nodeId, that should be failed or recovered")
    foundJobs.foreach(failOrRecoverAbandonedJob)
  }

  private def findAllJobs(): List[TaskJob[_]] = {
    val filters = new JobFilters()
    filters.node = null
    filters.withoutNode = false
    val jobs = jobService.findAllTaskJobs(filters, Pageable.unpaged(), ignoreUnknownType = true)
    jobs
  }

  private def findJobsForNode(nodeId: String): List[TaskJob[_]] = {
    val filters = new JobFilters()
    filters.node = nodeId
    if (null == nodeId) {
      filters.withoutNode = true
    }
    val jobs = jobService.findAllTaskJobs(filters, Pageable.unpaged(), ignoreUnknownType = true)
    jobs
  }

  private def failOrRecoverAbandonedJob(job: TaskJob[_]): Unit = {
    try {
      job.status match {
        case JobStatus.QUEUED =>
          // take over this job
          recover(job)
        case JobStatus.RUNNING if job.jobType == JobType.NEXT_CUSTOM_SCRIPT_TASK =>
          // old code has that for years, even if polling task crashed, we recover and execute it again
          recover(job)
        case JobStatus.RUNNING if job.jobType == JobType.CONTAINER_TASK =>
          // Container based task should be recovered by JobWorkerActor and JobExecutorActor
          logger.debug(s"Job ${job.id} of type ${job.jobType} for task ${job.taskId} will be recovered by JobWorkerActor.")
        case JobStatus.RUNNING =>
          logger.warn(s"Failing abandoned task ${job.taskId} of type ${job.jobType}, that was launched at ${job.startTime} on ${job.node}")
          releaseActorService.failTaskWithRetry(job.taskId, "Task failed. It was interrupted by a server restart.",
            User.SYSTEM, None, job.taskRef.getExecutionId)
          jobService.deleteByTaskIdAndExecutionId(job.taskId, job.taskRef.getExecutionId)
        case _ =>
          logger.warn(s"Unable to recover job $job. Please inspect it in 'Task Manager'.")
      }
    } catch {
      case t: Throwable =>
        logger.error(s"Error during recovery of job $job", t)
        Try(jobService.updateJobStatus(job, JobStatus.FAILED))
    }
  }

  /**
   * It's responsibility of a caller to ensure, that task is recoverable by using getRecoverableTaskIds
   */
  //noinspection ScalaStyle
  private def recover(jobToRecover: TaskJob[_]): Unit = {
    logger.debug(s"Recovering task '${jobToRecover.taskId}'")
    // job for this task should be already in the database, we only need to:
    // 1. change row in the database atomically so, that it is now assigned to *this* node instead
    // of dead node
    // 2. if change was successful -- then add it to our queue. Otherwise, it may mean, that
    // some other node already assigned that same to itself
    jobService.updateJobNodeAndStatus(jobToRecover, nodeId, JobStatus.QUEUED) match {
      case Some(jobRow: JobRow) =>
        jobToRecover.version = jobRow.version
        jobToRecover.node = jobRow.node
        jobToRecover.status = jobRow.status
        jobQueue.submitExisting(jobToRecover)
      case None =>
        logger.debug(s"Unable to re-assign abandoned job $jobToRecover, maybe it was already taken by another node")
        ()
    }
  }
}
