package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.{Ids, SSERepository}
import com.xebialabs.xlrelease.scheduler.logs.TaskLogCreated
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.SseEventSink


object CiSSEService {
  private val LOG_EVENT = "TASK_LOG_EVENT"
  private val COMMENT_EVENT = "TASK_COMMENT_EVENT"
  private val ATTACHMENT_EVENT = "TASK_ATTACHMENT_EVENT"
  private val TASK_STATUS_LINE = "TASK_STATUS_LINE"
  private val TASK_EXECUTION_EVENT = "TASK_EXECUTION_EVENT"
}

@Service
@EventListener
class CiSSEService @Autowired()(sseRepository: SSERepository) extends Logging {

  def followCi(taskId: String, sink: SseEventSink): Unit = {
    val folderLessId = Ids.getFolderlessId(taskId)
    sseRepository.addSink(folderLessId, sink)
  }

  @AsyncSubscribe
  def onEvent(event: DistributedXLReleaseEvent): Unit = {
    event match {
      case TaskLogCreated(taskId, executionId, _) =>
        sendEventToSink(taskId, CiSSEService.LOG_EVENT, executionId)
      case e: DistributedCommentEvent =>
        sendEventToSink(e.taskId, CiSSEService.COMMENT_EVENT, "")
      case e: DistributedAttachmentEvent =>
        sendEventToSink(e.containerId, CiSSEService.ATTACHMENT_EVENT, "")
      case e: DistributedTaskStatusLineUpdated =>
        sendEventToSink(e.taskId, CiSSEService.TASK_STATUS_LINE, e.statusLine)
      case e: DistributedTaskStatusEvent =>
        sendEventToSink(Ids.releaseIdFrom(e.containerId), CiSSEService.TASK_EXECUTION_EVENT, e.taskStatus.value())
      case _ =>
      // nothing to do yet
    }
  }

  private def sendEventToSink(taskId: String, eventName: String, message: AnyRef): Unit = {
    val folderLessId = Ids.getFolderlessId(taskId)
    val event = sseRepository.newEventBuilder().name(eventName).mediaType(MediaType.APPLICATION_JSON_TYPE).data(message.getClass, message).build()
    sseRepository.sendEventToSink(folderLessId, event)
  }

}
