package com.xebialabs.xlrelease.security

import com.google.common.base.Strings
import com.xebialabs.deployit.checks.Checks
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.deployit.security._
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.security.permission.PlatformPermissions.{ADMIN, EDIT_SECURITY}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{Release, Task, Team}
import com.xebialabs.xlrelease.principaldata.PrincipalDataProvider
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.security.XLReleasePermissions._
import com.xebialabs.xlrelease.security.XlrPermissionLabels._
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.service.{ArchivingService, CalendarService, TeamService}
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.views.TeamView
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import java.util
import java.util.function.{Consumer, Supplier}
import java.util.stream.Collectors.toList
import java.util.{Date, Optional, List => JList}
import scala.annotation.varargs
import scala.jdk.CollectionConverters._
import scala.jdk.StreamConverters._

object PermissionChecker {
  val GLOBAL_SECURITY_ALIAS = "global"

  private val folderAdminCannotAssignPermissionException = new PermissionDeniedException("You cannot assign this permission to a new or existing team.")
  private val folderAdminCannotEditTeamException = new PermissionDeniedException("You cannot edit a team with this assigned permission.")
  private val folderAdminCannotDeleteTeamException = new PermissionDeniedException("You cannot delete a team with this assigned permission.")
  private val userCannotManageTeamException = new PermissionDeniedException("You don't have the permissions required to manage this team.")
}

//noinspection ScalaStyle
@Component("permissionChecker")
class PermissionChecker @Autowired()(permissionEnforcer: PermissionEnforcer,
                                     releaseRepository: ReleaseRepository,
                                     taskRepository: TaskRepository,
                                     roleService: RoleService,
                                     teamService: TeamService,
                                     calendarService: CalendarService,
                                     securedCis: SecuredCis,
                                     archivingService: ArchivingService,
                                     xlrConfig: XlrConfig,
                                     principalDataProvider: PrincipalDataProvider) {

  def check(permission: Permission): Unit = {
    if (!hasGlobalPermission(permission)) {
      throw PermissionDeniedException.forPermission(permission, null.asInstanceOf[String])
    }
  }

  def check(permission: Permission, ciId: String): Unit = {
    if (!hasPermission(permission, ciId)) {
      throw PermissionDeniedException.forPermission(permission, ciId)
    }
  }

  def check(permission: Permission, release: Release): Unit = {
    if (!hasPermission(permission, release)) {
      throw PermissionDeniedException.forPermission(permission, release.getId)
    }
  }

  @varargs
  def checkAny(ciId: String, permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasPermission(permission, ciId)) {
        return
      }
    }
    throw exceptionForPermissions(ciId, permissions)
  }

  @varargs
  def checkAny(permissions: Permission*): Unit = {
    for (permission <- permissions) {
      if (hasGlobalPermission(permission)) return
    }
    throw exceptionForPermissions(permissions: _*)
  }


  @varargs
  private def exceptionForPermissions(permissions: Permission*) = {
    val exception = new PermissionDeniedException
    exception.add("You do not have %s permission on %s", permissions.mkString(" or "))
    exception
  }

  private def exceptionForPermissions(releaseId: String, permissions: Seq[Permission]) = {
    val exception = new PermissionDeniedException
    exception.add("You do not have %s permission on %s", permissions.mkString(" or "), releaseId)
    exception
  }

  def checkView(releaseId: String): Unit = checkAny(releaseId, AUDIT_ALL, getViewPermission(releaseId))

  def checkView(release: Release): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL)) {
      check(getViewPermission(release), release)
    }
  }

  private def canViewRelease(release: Release): Boolean = hasGlobalPermission(AUDIT_ALL) || hasCiPermission(getViewPermission(release), release)

  def canViewRelease(releaseId: String): Boolean = hasGlobalPermission(AUDIT_ALL) || hasCiPermission(getViewPermission(releaseId), releaseId)

  def checkEditAttachment(ciId: String): Unit = {
    if (Ids.isTaskId(ciId)) {
      checkIsAllowedToEditAttachmentsOnTask(ciId)
    } else {
      checkEdit(releaseIdFrom(ciId))
    }
  }

  def checkEdit(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE
    check(permissionToCheck, releaseId)
  }

  def checkEditTask(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  def checkEditDate(ciId: String): Unit = {
    if (isReleaseId(ciId) || Ids.isPhaseId(ciId)) {
      checkEdit(releaseIdFrom(ciId))
    } else {
      val releaseId = releaseIdFrom(ciId)
      val permissionsToCheck = if (releaseRepository.isTemplate(releaseId)) List(EDIT_TEMPLATE) else List(EDIT_RELEASE_TASK, EDIT_TASK_DATES)
      checkAny(releaseId, permissionsToCheck: _*)
    }
  }

  def canEditTask(releaseId: String): Boolean = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK
    hasPermission(permissionToCheck, releaseId)
  }

  def checkEditVariable(release: Release, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    if (hasPermission(permissionToCheck, release)) {
      return
    }
    for (task <- release.getAllTasks.asScala) {
      if ((hasPermissionToUpdateTask(task, release) || canEditTaskInputOutputProperties(release)) && task.getReferencedVariables.contains(variable)) {
        return
      }
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }

  def checkEditVariable(release: Release, task: Task, variable: Variable): Unit = {
    val permissionToCheck: Permission = if (release.isTemplate) EDIT_TEMPLATE else EDIT_RELEASE
    if (hasPermission(permissionToCheck, release)) {
      return
    }
    if ((hasPermissionToUpdateTask(task, release) || canEditTaskInputOutputProperties(release)) && task.getReferencedVariables.contains(variable)) {
      return
    }
    throw PermissionDeniedException.withMessage(s"You do not have edit variable permission on variable [${variable.getId}]")
  }

  def checkEditTaskConfigurationFacet(releaseId: String): Unit = {
    if (!(canEditTask(releaseId) || canEditTaskConfigurationFacet(releaseId))) {
      throw PermissionDeniedException.withMessage(s"You are not allowed to create facets on tasks of release [$releaseId]")
    }
  }

  def checkDeleteTasks(ids: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      checkEdit(releaseId)
    }
  }

  private def checkTasksFromSingleRelease(ids: JList[String]): Option[String] = {
    if (!ids.isEmpty) {
      val iterator: util.Iterator[String] = ids.iterator
      val releaseId: String = releaseIdFrom(iterator.next)
      while (iterator.hasNext) {
        if (!(releaseIdFrom(iterator.next) == releaseId)) {
          throw new IllegalArgumentException("You can not work with a set of tasks from different releases or templates")
        }
      }
      return Some(releaseId)
    }
    None
  }

  def checkReassignTasks(ids: JList[String], newUser: String): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(ids).orNull
    if (releaseId != null) {
      val release: Release = releaseRepository.findById(releaseId, ResolveOptions.WITH_DECORATORS)
      checkReassignTaskPermission(release)
      ids.forEach(id => this.checkReassignTaskToUser(id, newUser))
    }
  }

  def checkReassignTaskToUser(taskId: String, newUser: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkReassignTaskToUser(task, newUser)
  }

  def checkReassignTaskToUser(task: Task, newUser: String): Unit = {
    try {
      checkReassignTaskPermission(task.getRelease)
    } catch {
      case e: PermissionDeniedException =>
        if (!areUsersInTheSameTaskTeam(task, newUser)) {
          throw e
        }
    }
  }

  def areUsersInTheSameTaskTeam(task: Task, newUser: String): Boolean = {
    // REL-2554 members of a task's team can always assign the task to one another
    val currentAuthentication: Authentication = Permissions.getAuthentication
    if (task.hasTeam) {
      val teamOptional: Optional[Team] = teamService.findTeamByName(task.getRelease.getId, task.getTeam)
      if (teamOptional.isPresent) {
        val team: Team = teamOptional.get
        return isUserInTeam(currentAuthentication, team) && (newUser == null || isUserInTeam(newUser, team))
      }
    }
    false
  }

  def checkEditBlackoutPermission(release: Release): Unit = {
    if (release.isTemplate) {
      check(EDIT_TEMPLATE, release)
    } else {
      checkAny(release.getId, EDIT_RELEASE_TASK, EDIT_BLACKOUT)
    }
  }

  def checkEditFailureHandlerPermission(release: Release): Unit = {
    checkEditFailureHandlerPermission(release.getId, release.isTemplate)
  }

  def checkEditFailureHandlerPermission(releaseId: String): Unit = {
    checkEditFailureHandlerPermission(releaseId, releaseRepository.isTemplate(releaseId))
  }

  private def checkEditFailureHandlerPermission(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE_FAILURE_HANDLER, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_FAILURE_HANDLER, EDIT_RELEASE_TASK)
    }
  }

  def checkEditPreconditionPermission(release: Release): Unit = {
    checkEditPreconditionPermission(release.getId, release.isTemplate)
  }

  def checkEditPreconditionPermission(releaseId: String): Unit = {
    checkEditPreconditionPermission(releaseId, releaseRepository.isTemplate(releaseId))
  }

  private def checkEditPreconditionPermission(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE_PRECONDITION, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_PRECONDITION, EDIT_RELEASE_TASK)
    }
  }

  private def isUserInTeam(authentication: Authentication, team: Team): Boolean = isUserInTeam(authentication.getName, team)

  private def isUserInTeam(user: String, team: Team): Boolean =
    team.hasMember(user) || team.hasAnyRole(roleService.getRolesFor(user)) || isExternalUserPrincipalIsFromTheGroupAssigned(user, team)

  private def isExternalUserPrincipalIsFromTheGroupAssigned(user: String, team: Team): Boolean = {
    val authorities = principalDataProvider.getAuthorities(user).asScala
    val principalsOfUser: Set[String] = authorities.map(_.getAuthority).toSet ++ Set(user)
    team.getRoles.asScala.map(role => roleService.getRoleForRoleName(role).getPrincipals.asScala).flatMap(_.toList).toSet.exists(principalsOfUser.contains)
  }

  def checkReassignTaskPermission(releaseId: String): Unit = {
    checkEditOrReassignTask(releaseId, releaseRepository.isTemplate(releaseId))
  }

  def checkReassignTaskPermission(release: Release): Unit = {
    checkEditOrReassignTask(release.getId, release.isTemplate)
  }

  private def checkEditOrReassignTask(releaseId: String, isTemplate: Boolean): Unit = {
    if (isTemplate) {
      check(EDIT_TEMPLATE, releaseId)
    } else {
      checkAny(releaseId, EDIT_RELEASE_TASK, REASSIGN_RELEASE_TASK)
    }
  }

  def checkReopenTaskInRelease(releaseId: String): Unit = {
    check(VIEW_RELEASE, releaseId)
    checkAny(releaseId, EDIT_RELEASE_TASK, TASK_TRANSITION)
  }

  def checkReopenTasksInRelease(taskIds: JList[String]): Unit = {
    val releaseId: String = checkTasksFromSingleRelease(taskIds).orNull
    checkReopenTaskInRelease(releaseId)
  }

  /**
   * Checks whether user can add, or edit comments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToCommentOnTask(taskId: String): Unit = {
    try {
      if (!isCurrentUserAdmin) {
        checkView(releaseIdFrom(taskId))
      }
    } catch {
      case e: NotFoundException =>
        if (archivingService.exists(taskId)) {
          throw PermissionDeniedException.withMessage(s"You cannot comment on archived task [$taskId]")
        } else {
          throw e
        }
    }
  }

  /**
   * Checks whether user can add or delete attachments on task.
   * Throws {@link PermissionDeniedException} if that is not allowed.
   */
  def checkIsAllowedToEditAttachmentsOnTask(taskId: String): Unit = {
    try {
      val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
      if (!hasPermissionToUpdateTask(task, task.getRelease) && !hasPermission(EDIT_RELEASE_TASK_ATTACHMENT, task.getRelease)) {
        throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Task owner', 'Member of task team', 'edit task' or 'edit task attachments'")
      }
    } catch {
      case e: NotFoundException =>
        if (taskRepository.exists(taskId)) {
          throw PermissionDeniedException.withMessage(s"You cannot upload attachments on archived task [$taskId]")
        } else {
          throw e
        }
    }
  }

  def filterAllowedToCommentOnTasks(taskIds: JList[String]): JList[String] = filterTasksSilently(taskIds, this.checkIsAllowedToCommentOnTask)

  def checkIsAllowedToStartTask(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPostponedDueToBlackout) {
      canOverrideBlackout(task)
    } else {
      if (task.isDelayDuringBlackout && calendarService.isInBlackout(new Date)) {
        canOverrideBlackout(task)
      }
      checkTaskTransitionPermission(taskId)
    }
  }

  private def canOverrideBlackout(task: Task): Unit = {
    if (!hasCiPermission(EDIT_BLACKOUT, task.getRelease) && !hasCiPermission(EDIT_RELEASE_TASK, task.getRelease)) {
      throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Edit task blackout'")
    }
  }

  def checkIsAllowedToWorkOnTask(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    checkHasPermissionsToUpdateTask(task)
  }

  def checkTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (!hasTaskTransitionPermission(task)) {
      throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
    }
  }

  def checkAdvanceTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPlanned && !hasAdvanceTaskTransitionPermission(task)) {
      throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
    }
  }

  def checkRelevantTaskTransitionPermission(taskId: String): Unit = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    checkTaskIsUpdatable(task)
    if (task.isPlanned) {
      if (!hasAdvanceTaskTransitionPermission(task)) {
        throw PermissionDeniedException.withMessage(s"You cannot make advance task transition for task=[$taskId]. You need to have permission or be release owner")
      }
    } else {
      if (!hasTaskTransitionPermission(task)) {
        throw PermissionDeniedException.withMessage(s"You cannot make task transition for task=[$taskId]. You need to have permission or be either release owner or task owner")
      }
    }
  }

  def filterTasksWithTaskTransitionPermission(taskIds: JList[String]): JList[String] = {
    def filterTasks(id: String): Boolean = {
      val task: Task = taskRepository.findById(id, ResolveOptions.WITH_DECORATORS)
      hasTaskTransitionPermission(task) && task.isUpdatable
    }

    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) => taskIds.asScala.filter(filterTasks).toList.asJava
      case None => new util.ArrayList[String]
    }
  }

  private def hasTaskTransitionPermission(task: Task): Boolean = {
    val release: Release = task.getRelease
    val allowReleaseOwnerTaskTransition = xlrConfig.isReleaseOwnerTaskTransitionAllowed
    (allowReleaseOwnerTaskTransition && release.hasOwner(Permissions.getAuthenticatedUserName)) || hasPermissionToUpdateTask(task, null) || hasCiPermission(TASK_TRANSITION, release)
  }

  private def hasAdvanceTaskTransitionPermission(task: Task): Boolean = {
    val release: Release = task.getRelease
    isCurrentUserAdmin || canEditTask(release.getId) || hasCiPermission(ADVANCE_TASK_TRANSITION, release)
  }

  def isAllowedToWorkOnTask(taskId: String): Boolean = {
    val task: Task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS)
    task.isUpdatable && hasPermissionToUpdateTask(task, null)
  }

  def filterStartableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => check(START_RELEASE, id))

  def filterAbortableReleases(releaseIds: JList[String]): JList[String] = filterSilently(releaseIds, (id: String) => check(ABORT_RELEASE, id))

  private def checkTaskIsUpdatable(task: Task): Unit = {
    if (!task.isUpdatable) {
      throw new IllegalArgumentException("You can not work with a defunct or done in advance task")
    }
  }

  def filter(items: JList[Release], permission: Permission): JList[Release] = items.asScala.filter(hasPermission(permission, _)).toList.asJava

  private[security] def filterViewables(configurationItems: JList[ConfigurationItem]): JList[ConfigurationItem] = {
    def isGranted(configurationItem: ConfigurationItem): Boolean =
      configurationItem.isInstanceOf[Release] && canViewRelease(configurationItem.asInstanceOf[Release])

    configurationItems.asScala.filter(isGranted).toList.asJava
  }

  private def checkPermission(permission: Permission, targetId: String): Unit = {
    if (!hasCiPermission(permission, targetId)) {
      throw PermissionDeniedException.forNodeAndPrivilege(targetId, s"'${permission.label()}'")
    }
  }

  def checkIsAllowedToCreateReleaseFromTemplate(templateId: String, targetFolderId: String): Unit = {
    if (isNullId(targetFolderId)) {
      throw new IllegalArgumentException("targetFolderId can not be null")
    } else if (!isFolderId(targetFolderId) && !ROOT_FOLDER_ID.equals(targetFolderId)) {
      throw new IllegalArgumentException(s"targetFolderId['$targetFolderId'] is not a valid folder id")
    }

    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    checkPermission(VIEW_TEMPLATE, templateId)
    val sourceFolderId = findFolderId(templateId)
    val isRootTemplate = !isInFolder(templateId)

    if (isRootTemplate) {
      if (ROOT_FOLDER_ID.equals(targetFolderId)) {
        val hasCreateReleasePermission = hasGlobalPermission(CREATE_RELEASE) || hasPermission(CREATE_RELEASE_FROM_TEMPLATE, templateId)
        if (!hasCreateReleasePermission) {
          throw PermissionDeniedException.forNodeAndPrivilege(templateId, s"'${CREATE_RELEASE.label()}' (global or on the template)")
        }
      } else {
        checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
        checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, templateId)
      }
    } else {
      checkPermission(CREATE_RELEASE_FROM_TEMPLATE, targetFolderId)
      if (targetFolderId != sourceFolderId) {
        checkPermission(CREATE_RELEASE_IN_ANOTHER_FOLDER, sourceFolderId)
      }
    }
  }

  /**
   * Used to check create release permission when release is created without template
   */
  def checkIsAllowedToCreateReleaseInFolder(folderId: String): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    if (!Strings.isNullOrEmpty(folderId) && isFolderId(folderId) && !isRoot(folderId)) {
      check(CREATE_RELEASE_FROM_TEMPLATE, folderId)
    } else {
      check(CREATE_RELEASE)
    }
  }

  def checkIsAllowedToRegisterRunner(): Unit = {
    checkAuthenticated()
    if (isCurrentUserAdmin) {
      return
    }
    check(XLReleasePermissions.RUNNER_REGISTRATION)
  }

  def checkEditSecurity(releaseId: String): Unit = {
    if (hasGlobalPermission(EDIT_SECURITY)) {
      return
    }
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) EDIT_TEMPLATE_SECURITY else EDIT_RELEASE_SECURITY
    check(permissionToCheck, releaseId)
  }

  private def hasCiPermission(permission: Permission, release: Release): Boolean = {
    if (!isPermissionApplicableTo(permission, release.getId)) {
      return false
    }
    if (release.isArchived) {
      return hasArchivedReleasePermission(permission, () => release)
    }
    permissionEnforcer.hasLoggedInUserPermission(permission, securedCis.getEffectiveSecuredCi(release.getId).getSecurityUid)
  }

  private def hasCiPermission(permission: Permission, containerId: String): Boolean = {
    if (!isPermissionApplicableTo(permission, containerId)) {
      return false
    }
    if (isReleaseId(containerId) && !releaseRepository.exists(containerId) && archivingService.exists(containerId)) {
      return hasArchivedReleasePermission(permission, () => archivingService.getRelease(containerId))
    }
    permissionEnforcer.hasLoggedInUserPermission(permission, securedCis.getEffectiveSecuredCi(containerId).getSecurityUid)
  }

  private def hasArchivedReleasePermission(permission: Permission, release: Supplier[Release]): Boolean = {
    if (hasGlobalPermission(ADMIN)) {
      return true
    }
    val auth: Authentication = getAuthentication
    val userRoles: JList[Role] = roleService.getRolesFor(auth)
    val userPrincipals: util.Collection[String] = Permissions.authenticationToPrincipals(auth)
    release.get.getPermissions(userPrincipals, userRoles).contains(permission.getPermissionName)
  }

  def hasGlobalPermission(permission: Permission): Boolean = {
    checkAuthenticated()
    isPermissionApplicableTo(permission, PermissionChecker.GLOBAL_SECURITY_ALIAS) && permissionEnforcer.hasLoggedInUserPermission(permission)
  }

  def hasPermission(permission: Permission, ciId: String): Boolean = hasGlobalPermission(permission) || hasCiPermission(permission, ciId)

  def hasPermission(permission: Permission, release: Release): Boolean = hasGlobalPermission(permission) || hasCiPermission(permission, release)

  def checkHasPermissionsToUpdateTask(task: Task): Unit = {
    if (!hasPermissionToUpdateTask(task, null)) {
      throw PermissionDeniedException.forNodeAndPrivilege(task.getId, "'Task owner', 'Member of task team' or 'edit task'")
    }
  }

  private def hasPermissionToUpdateTask(task: Task, release: Release): Boolean = {
    if (isCurrentUserAdmin || owns(task)) {
      return true
    }

    val resolvedRelease = if (release == null) releaseRepository.findById(releaseIdFrom(task.getId), ResolveOptions.WITH_DECORATORS) else release
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(resolvedRelease.getId)) EDIT_TEMPLATE else EDIT_RELEASE_TASK

    if (hasCiPermission(permissionToCheck, resolvedRelease)) {
      return true
    }
    if (task.hasTeam) {
      return isMemberOrRoleOf(releaseIdFrom(task.getId), task.getTeam)
    }
    false
  }

  private def canEditTaskInputOutputProperties(release: Release): Boolean = hasPermission(EDIT_RELEASE_TASK_CONFIGURATION, release)

  private def canEditTaskConfigurationFacet(releaseId: String): Boolean = hasPermission(EDIT_RELEASE_TASK_CONFIGURATION_FACET, releaseId)

  def checkViewTask(task: Task): Unit = {
    if (!hasGlobalPermission(AUDIT_ALL) && !hasViewTaskPermissions(task, Permissions.getAuthenticatedUserName, null)) {
      throw new PermissionDeniedException(s"You are not allowed to view task [${task.getId}]")
    }
  }

  def checkViewFolder(containerId: String): Unit = {
    if (containerId != ROOT_FOLDER_ID && !hasGlobalPermission(AUDIT_ALL)) {
      check(VIEW_FOLDER, containerId)
    }
  }

  /**
   * A user can see a task only if he meets at least one of those 4 conditions:
   * - he has AUDIT_ALL global permission
   * - he has VIEW_PERMISSION on the release or template the task belongs to
   * - he is the owner of the task
   * - he belongs to a team assigned to the task
   */
  private[security] def hasViewTaskPermissions(task: Task, username: String, userRoles: JList[Role]): Boolean = {
    val releaseId: String = releaseIdFrom(task.getId)
    (hasGlobalPermission(AUDIT_ALL)
      || hasPermission(getViewPermission(releaseId), releaseId)
      || task.hasOwner(username)
      || isMemberOrRoleOf(releaseId, task.getTeam, userRoles))
  }

  def isMemberOrRoleOf(releaseId: String, teamName: String): Boolean = isMemberOrRoleOf(releaseId, teamName, null)

  private def isMemberOrRoleOf(releaseId: String, teamName: String, prefetchedUserRoles: JList[Role]): Boolean = {
    if (!StringUtils.hasText(teamName)) {
      return false
    }
    val teamsToCheck: Seq[Team] = teamService.findTeamsByNames(releaseId, util.Arrays.asList(teamName, Team.RELEASE_ADMIN_TEAMNAME)).toScala(Seq)
    val currentUser: String = Permissions.getAuthenticatedUserName
    if (teamsToCheck.exists(_.hasMember(currentUser))) {
      return true
    }
    val roles: JList[Role] = if (prefetchedUserRoles == null) roleService.getRolesFor(getAuthentication) else prefetchedUserRoles
    teamsToCheck.exists(_.hasAnyRole(roles))
  }

  private def getViewPermission(release: Release): Permission = if (release.isTemplate) VIEW_TEMPLATE else VIEW_RELEASE

  private def owns(task: Task): Boolean = task.hasOwner(Permissions.getAuthenticatedUserName)

  private def getViewPermission(releaseId: String): Permission = if (releaseRepository.isTemplate(releaseId)) VIEW_TEMPLATE else VIEW_RELEASE

  private def checkAuthenticated(): Unit = {
    if (isNotAuthenticated) {
      throw new AuthenticationFailureException("Authentication is missing. Did you specify 'Run automated tasks as user' property of the release?")
    }
  }

  def isNotAuthenticated: Boolean = getAuthentication == null

  def isCurrentUserAdmin: Boolean = permissionEnforcer.isCurrentUserAdmin

  private def isPermissionApplicableTo(permission: Permission, containerId: String): Boolean = permission != null && permission.isApplicableTo(containerId)

  private def filterTasksSilently(taskIds: JList[String], checker: Consumer[String]): JList[String] = {
    checkTasksFromSingleRelease(taskIds) match {
      case Some(_) => filterSilently(taskIds, checker)
      case None => new util.ArrayList[String]
    }
  }

  def filterSilently(ids: JList[String], checker: Consumer[String]): JList[String] = {
    def filterId(id: String): Boolean = {
      try {
        checker.accept(id)
        true
      } catch {
        case _: Exception => false
      }
    }

    ids.asScala.filter(filterId).toList.asJava
  }

  def checkCopyTask(releaseId: String): Unit = {
    checkView(releaseId)
    checkEdit(releaseId)
    checkEditTask(releaseId)
  }

  def copyPhase(releaseId: String): Unit = {
    checkView(releaseId)
    checkEdit(releaseId)
  }

  def checkLockTaskPermission(releaseId: String): Unit = {
    val permissionToCheck: Permission = if (releaseRepository.isTemplate(releaseId)) LOCK_TEMPLATE_TASK else LOCK_RELEASE_TASK
    check(permissionToCheck, releaseId)
  }

  def checkViewTeams(teamContainerId: String): Unit = {
    if (isReleaseId(teamContainerId)) {
      checkEdit(teamContainerId)
    } else {
      if (isFolderId(teamContainerId)) {
        checkAny(teamContainerId, EDIT_FOLDER_SECURITY, EDIT_FOLDER_TEAMS, EDIT_FOLDER_NOTIFICATIONS,
          EDIT_DELIVERY_PATTERN,
          EDIT_RELEASE_DELIVERY,
          VIEW_DELIVERY_PATTERN,
          VIEW_RELEASE_DELIVERY)
      }
      else {
        throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
      }
    }
  }

  def checkEditTeams(teamContainerId: String, teams: JList[TeamView]): Unit = {
    checkFolderTeams(teamContainerId, () => checkFolderAdmin(teamContainerId, () => {
      val toUpdate: Seq[Team] = withoutUnchangedTeams(teamContainerId, teams.asScala.map(_.toTeam).toSeq)
      toUpdate.foreach(team => canUpdateTeam(teamContainerId, team))
    }))
  }

  def checkDeleteTeams(teamContainerId: String, teamIds: JList[String]): Unit = {
    checkFolderTeams(teamContainerId, () => checkFolderAdmin(teamContainerId, () => teamIds.forEach((teamId: String) => canDeleteTeam(teamContainerId, teamId))))
  }

  def getUserFolderTeams(folderId: String): JList[Team] = {
    val username: String = User.AUTHENTICATED_USER.getName
    val folderTeams: JList[Team] = teamService.getEffectiveTeams(folderId)
    val userRoles: JList[Role] = roleService.getRolesFor(Permissions.getAuthentication)
    folderTeams.stream.filter((team: Team) => team.hasMember(username) || team.hasAnyRole(userRoles)).collect(toList())
  }

  def checkDeleteOwnTeams(teamContainerId: String): Unit = {
    checkFolderTeams(teamContainerId, () => checkFolderAdmin(teamContainerId, () => getUserFolderTeams(teamContainerId).forEach(this.canDeleteTeam)))
  }

  private def checkFolderTeams(teamContainerId: String, checkFolder: Runnable): Unit = {
    if (isReleaseId(teamContainerId)) {
      checkEdit(teamContainerId)
    } else if (isFolderId(teamContainerId)) {
      checkFolder.run()
    } else {
      throw new Checks.IncorrectArgumentException("[%s] is not a release or a folder", teamContainerId)
    }
  }

  private def checkFolderAdmin(containerId: String, checkFolderAdmin: Runnable): Unit = {
    val superUser: Boolean = isCurrentUserAdmin
    val owner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, containerId)
    val admin: Boolean = hasPermission(EDIT_FOLDER_TEAMS, containerId)
    if (admin && !owner && !superUser) {
      checkFolderAdmin.run()
    } else if (!superUser && !owner) {
      throw PermissionChecker.userCannotManageTeamException
    }
  }

  private def canDeleteTeam(containerId: String, teamId: String): Unit = {
    val folderTeams: JList[Team] = teamService.getEffectiveTeams(containerId)
    val maybeTeamToDelete: Optional[Team] = folderTeams.stream.filter((team: Team) => team.getId == teamId).findAny
    maybeTeamToDelete.ifPresent(this.canDeleteTeam)
  }

  private def canDeleteTeam(team: Team): Unit = {
    if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
      throw PermissionChecker.folderAdminCannotDeleteTeamException
    }
  }

  private def withoutUnchangedTeams(containerId: String, teams: Seq[Team]): Seq[Team] = {
    val folderTeams: JList[Team] = teamService.getEffectiveTeams(containerId)
    teams.filter(!folderTeams.contains(_))
  }

  private def canUpdateTeam(containerId: String, team: Team): Unit = {
    val oldTeam: Optional[Team] = teamService.getEffectiveTeams(containerId).stream.filter((t: Team) => t.getTeamName == team.getTeamName).findFirst
    if (team.isFolderOwnerTeam || team.isFolderAdminTeam) {
      throw PermissionChecker.folderAdminCannotAssignPermissionException
    }

    val owner: Boolean = hasPermission(EDIT_FOLDER_SECURITY, containerId)
    if (team.hasPermission(EDIT_FOLDER.getPermissionName) && !owner) {
      throw PermissionChecker.folderAdminCannotAssignPermissionException
    }

    oldTeam.ifPresent((old: Team) => {
      if (old.isFolderOwnerTeam || old.isFolderAdminTeam) {
        throw PermissionChecker.folderAdminCannotEditTeamException
      }
    })
  }

  def checkEditNotification(folderId: String): Unit = if (folderId == null) check(ADMIN) else check(EDIT_FOLDER_NOTIFICATIONS, folderId)
}
