package com.xebialabs.xlrelease.scheduler.logs

import akka.actor.{Actor, NoSerializationVerificationNeeded, ReceiveTimeout}
import akka.event.LoggingReceive
import akka.event.slf4j.SLF4JLogging
import com.xebialabs.xlrelease.repository.SSERepository
import com.xebialabs.xlrelease.scheduler.logs.ExecutionLogWatchActor._
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.akka.spring.SpringActor
import org.apache.commons.io.IOUtils

import java.net.URI
import java.time.{Duration, Instant}
import javax.ws.rs.sse.{OutboundSseEvent, SseEventSink}
import scala.concurrent.duration.DurationInt
import scala.util.Using

@SpringActor
class ExecutionLogWatchActor(taskExecutionLogService: TaskExecutionLogService, storageService: StorageService, sseRepository: SSERepository)
  extends Actor with SLF4JLogging {
  var executionId: String = _
  var taskId: String = _
  var lastEventTimestamp: Instant = Instant.now()

  context.setReceiveTimeout(5.seconds)

  override def receive: Receive = LoggingReceive {
    case m: WatcherMsg => handleWatchMsgs(m)
    case ReceiveTimeout => self ! Check(executionId)
  }

  override def postStop(): Unit = {
    sseRepository.closeAll(executionId)
  }

  //noinspection ScalaStyle
  private def handleWatchMsgs(watcherMsg: WatcherMsg): Unit = watcherMsg match {
    case StartWatch(taskId, executionId, sseEventSink) =>
      this.executionId = executionId
      this.taskId = taskId
      sseRepository.addSink(executionId, sseEventSink)
      streamCurrentState(executionId)
      streamLastChunk(executionId)
      self ! Check(executionId)
    case StopWatching(executionId) =>
      sseRepository.closeAll(executionId)
      self ! Check(executionId)
    case Check(executionId) =>
      val openSinks = sseRepository.get(executionId)
      // should unsubscribe... any msg that arrives will be sent to dead letter
      if (openSinks.isEmpty) {
        context.stop(self)
      } else {
        val maybeExecutionEntry = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId)
        val executionIsCompletedOrMissing = maybeExecutionEntry.forall(_.endDate != null)
        if (executionIsCompletedOrMissing) {
          self ! StopWatching(executionId)
        } else {
          // if there was no new entry for certain amount of time ... send ping so we would close closed sinks
          val durationSinceLastEvent = Duration.between(lastEventTimestamp, Instant.now()).toSeconds
          if (durationSinceLastEvent > 10 && durationSinceLastEvent < 60) {
            val ping = sseRepository.newEventBuilder().name(PING_SSE_EVENT).data(new String(":\n\n")).build()
            sendEventToAllSinks(executionId, ping)
          } else if (durationSinceLastEvent >= 60) {
            self ! StopWatching(executionId)
          }
        }
      }
    case NewEntry(executionId, newEntryUri) =>
      this.lastEventTimestamp = Instant.now()
      // store instant of the entry into internal state
      val payload = Using.resource(storageService.get(JobEntryRef(newEntryUri))) { content =>
        IOUtils.toByteArray(content)
      }
      val event = sseRepository.newEventBuilder().name(LOG_CHUNK_ENTRY_CREATED_EVENT).data(new String(payload)).build()
      sendEventToAllSinks(executionId, event)
  }

  private def sendEventToAllSinks(executionId: String, event: OutboundSseEvent): Unit = {
    sseRepository.sendEventToSink(executionId, event)
  }

  private def streamCurrentState(executionId: String): Unit = {
    val (status, jobId, chunk) = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId) match {
      case Some(row) => (if (row.endDate == null) "in_progress" else "closed", row.lastJob, row.lastChunk)
      case None => ("unknown", -1, -1)
    }
    val payload = s"$status, $jobId, $chunk"
    val statusEvent = sseRepository.newEventBuilder().name(EXECUTION_LOG_STATUS_EVENT).data(new String(payload)).build()
    sseRepository.sendEventToSink(executionId, statusEvent)

  }

  private def streamLastChunk(executionId: String): Unit = {
    val maybeRow = taskExecutionLogService.getTaskExecutionEntry(taskId, executionId)
    maybeRow.foreach { row =>
      // row -> log entry ref
      val taskIdHash = row.taskIdHash
      val executionId = row.executionId
      val jobId = row.lastJob
      val chunk = row.lastChunk
      val uriPath = s"/jobs/$taskIdHash/$executionId/$jobId/$chunk"
      val uriScheme = URI_SCHEME_LOCAL_STORAGE
      val entryUri = URI.create(s"$uriScheme://$uriPath")
      val payload = Using.resource(storageService.get(JobEntryRef(entryUri))) { content =>
        IOUtils.toByteArray(content)
      }
      val event = sseRepository.newEventBuilder().name(LOG_CHUNK_ENTRY_CREATED_EVENT).data(new String(payload)).build()
      sseRepository.sendEventToSink(executionId, event)
    }
  }

}

object ExecutionLogWatchActor {

  final val LOG_CHUNK_ENTRY_CREATED_EVENT: String = "log-chunk-entry-created"

  final val PING_SSE_EVENT: String = "ping"

  final val EXECUTION_LOG_STATUS_EVENT: String = "execution-status" // in_progress or closed, last job_id and last_chunk?

  sealed trait WatcherMsg {
    def executionId: String
  }

  case class StartWatch(taskId: String, executionId: String, sink: SseEventSink) extends WatcherMsg with NoSerializationVerificationNeeded

  case class StopWatching(executionId: String) extends WatcherMsg

  case class Check(executionId: String) extends WatcherMsg

  case class NewEntry(executionId: String, newEntryUri: URI) extends WatcherMsg
}
