package com.xebialabs.xlrelease.repository

import grizzled.slf4j.Logging
import org.jboss.resteasy.plugins.providers.sse.OutboundSseEventImpl
import org.springframework.stereotype.Repository

import java.util.concurrent.ConcurrentHashMap
import javax.ws.rs.sse.{OutboundSseEvent, SseEventSink}
import scala.collection.{concurrent, mutable}
import scala.jdk.CollectionConverters.{ConcurrentMapHasAsScala, SetHasAsScala}
import scala.jdk.FutureConverters.CompletionStageOps
import scala.util.Try

trait SSERepository {

  def get(id: String): Set[SseEventSink]

  def addSink(id: String, sink: SseEventSink): Unit

  def sendEventToSink(id: String, event: OutboundSseEvent): Unit

  def closeAll(id: String): Unit

  def newEventBuilder() = new OutboundSseEventImpl.BuilderImpl()

}

@Repository
class InMemorySSERepository extends SSERepository with Logging {

  private val sessions: concurrent.Map[String, mutable.Set[SseEventSink]] = new ConcurrentHashMap[String, mutable.Set[SseEventSink]].asScala

  private def createConcurrentSet(): mutable.Set[SseEventSink] = {
    java.util.Collections.newSetFromMap(new ConcurrentHashMap[SseEventSink, java.lang.Boolean]).asScala
  }

  override def get(id: String): Set[SseEventSink] = {
    removeClosedSinks()
    sessions.getOrElse(id, createConcurrentSet()).toSet
  }

  override def sendEventToSink(id: String, event: OutboundSseEvent): Unit = {
    val sinks = get(id)
    sinks.foreach(sink => {
      if (!sink.isClosed) {
        sink.send(event).asScala.recover { e =>
          logger.debug("can't send SSE event", e)
          sink.close()
        }(scala.concurrent.ExecutionContext.parasitic)
      }
    })
  }

  override def addSink(id: String, sink: SseEventSink): Unit = {
    removeClosedSinks()
    val taskSinks = sessions.getOrElse(id, createConcurrentSet())
    sessions.put(id, taskSinks += sink)
  }


  override def closeAll(id: String): Unit = {
    val sinks = get(id)
    sinks.foreach(s => Try(s.close()))
    removeClosedSinks()
  }

  private def removeClosedSinks(): Unit = {
    sessions.values.foreach { set => set.filterInPlace(!_.isClosed) }
    sessions.filterInPlace { (_, set) => set.nonEmpty }
  }
}

