package com.xebialabs.xlrelease.api.v1.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.repository.WorkDirContext;
import com.xebialabs.deployit.repository.WorkDirFactory;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.ReleaseApi;
import com.xebialabs.xlrelease.api.v1.TeamFacade;
import com.xebialabs.xlrelease.api.v1.VariableComponent;
import com.xebialabs.xlrelease.api.v1.forms.AbortRelease;
import com.xebialabs.xlrelease.api.v1.forms.ReleaseOrderMode;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Attachment;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.Page;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.search.ReleaseCountResults;
import com.xebialabs.xlrelease.search.ReleaseFullSearchResult;
import com.xebialabs.xlrelease.search.ReleaseSearchResult;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.variable.VariableHelper;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.io.ByteStreams.copy;
import static com.google.common.io.ByteStreams.toByteArray;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.ABORT_RELEASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.RESTART_PHASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.START_RELEASE;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static java.lang.String.format;

@Controller
public class ReleaseApiImpl implements ReleaseApi {

    private static final Logger logger = LoggerFactory.getLogger(ReleaseApiImpl.class);

    private final PermissionChecker permissions;
    private final ReleaseService releaseService;
    private final ReleaseSearchService releaseSearchService;
    private final ReleaseActorService releaseActorService;
    private final VariableComponent variableComponent;
    private final TeamFacade teamFacade;
    private final AttachmentService attachmentService;
    private final VariableService variableService;
    private final FolderVariableService folderVariableService;
    private final PhaseService phaseService;

    @Autowired
    public ReleaseApiImpl(PermissionChecker permissions,
                      ReleaseService releaseService,
                      ReleaseSearchService releaseSearchService,
                      VariableComponent variableComponent,
                      ReleaseActorService releaseActorService,
                      TeamFacade teamFacade,
                      AttachmentService attachmentService,
                      VariableService variableService,
                      FolderVariableService folderVariableService,
                      PhaseService phaseService) {
        this.permissions = permissions;
        this.releaseService = releaseService;
        this.releaseSearchService = releaseSearchService;
        this.releaseActorService = releaseActorService;
        this.variableComponent = variableComponent;
        this.teamFacade = teamFacade;
        this.attachmentService = attachmentService;
        this.variableService = variableService;
        this.folderVariableService = folderVariableService;
        this.phaseService = phaseService;
    }

    @Timed
    @Override
    public Response downloadAttachment(String attachmentId) {
        permissions.checkView(releaseIdFrom(attachmentId));
        Attachment attachment = attachmentService.findByIdIncludingArchived(attachmentId);

        // Streaming output write method will be called AFTER WorkDirInjectorFactory executes finally block
        //  and @Workdir annotation will always clear context after it invokes surrounding method.
        //  This means that @Workdir annotation will have no effect and will actually make things worse.
        StreamingOutput output = outputStream -> {
            WorkDirContext.initWorkdir(WorkDirFactory.DOWNLOAD_WORKDIR_PREFIX);
            // workdir has to be created HERE if it's cleaned up ALWAYS (clean attribute on the @WorkDir annotation)
            try (InputStream inputStream = attachment.getFile().getInputStream()) {
                copy(inputStream, outputStream);
            } finally {
                outputStream.flush();
                // WORKDIR has to be cleaned up here
                WorkDirContext.get().delete();
                WorkDirContext.clear();
            }
        };
        String fileName = attachment.getFile().getName();

        return ResponseHelper.streamFile(fileName, output, attachment.getContentType());
    }

    @Timed
    @Override
    public byte[] getAttachment(String attachmentId) throws IOException {
        permissions.checkView(releaseIdFrom(attachmentId));

        WorkdirHolder.initWorkdir(WorkDirFactory.DOWNLOAD_WORKDIR_PREFIX);
        Attachment attachment = attachmentService.findByIdIncludingArchived(attachmentId);

        final WorkDir workDir = WorkDirContext.get();

        byte[] byteArray;
        try (InputStream inputStream = attachment.getFile().getInputStream()) {
            byteArray = toByteArray(inputStream);
        } finally {
            workDir.delete();
        }

        return byteArray;
    }

    @Timed
    @Override
    public ReleaseCountResults countReleases(ReleasesFilters releasesFilters) {
        if (null == releasesFilters) {
            releasesFilters = new ReleasesFilters();
        }
        return releaseSearchService.count(releasesFilters);
    }

    @Timed
    @Override
    public List<Release> searchReleases(ReleasesFilters releasesFilters, Long page, Long resultsPerPage, Boolean pageIsOffset) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than 100");

        if (null == releasesFilters) {
            releasesFilters = new ReleasesFilters();
        }
        return releaseSearchService.search(releasesFilters, page, resultsPerPage, pageIsOffset).getReleases();
    }

    @Timed
    @Override
    public List<Release> searchReleases(ReleasesFilters releasesFilters, Long page, Long resultsPerPage) {
        return searchReleases(releasesFilters, page, resultsPerPage, false);
    }

    @Timed
    @Override
    public List<Release> searchReleases(ReleasesFilters releasesFilters) {
        return searchReleases(releasesFilters, DEFAULT_PAGE, DEFAULT_RESULTS_PER_PAGE);
    }

    @Timed
    @Override
    public ReleaseFullSearchResult fullSearchReleases(Long page, Long archivePage, Long resultsPerPage, Long archiveResultsPerPage, ReleasesFilters releasesFilters) {
        Optional<Object> size = Optional.ofNullable(resultsPerPage);
        Page pageForCurrent = Page.parse(Optional.ofNullable(page), size, Optional.empty());

        Optional<Object> archiveSize = Optional.<Object>ofNullable(archiveResultsPerPage).map(Optional::of).orElse(size);
        Optional<Object> archiveOrCurrentPage = Optional.<Object>ofNullable(archivePage).map(Optional::of).orElse(Optional.ofNullable(page));
        Page pageForArchive = Page.parse(archiveOrCurrentPage, archiveSize, Optional.empty());

        return releaseSearchService.fullSearch(releasesFilters, pageForCurrent, pageForArchive);
    }

    @Timed
    @Override
    public List<Release> getReleases(Long page, Long resultsPerPage, Integer depth) {
        checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than 100");
        ReleasesFilters filters = new ReleasesFilters();
        filters.setPlanned(true);
        filters.setActive(true);
        filters.setOrderBy(ReleaseOrderMode.start_date);
        ReleaseSearchResult searchResult = releaseSearchService.search(filters, page, resultsPerPage, depth);
        return searchResult.getReleases();
    }

    @Timed
    @Override
    public List<Release> getReleases() {
        // keep old behavior
        return getReleases(DEFAULT_PAGE, DEFAULT_RESULTS_PER_PAGE, Integer.MAX_VALUE);
    }

    @Timed
    @Override
    public Release getRelease(String releaseId, boolean withRoleIds) {
        Release release = releaseService.findById(releaseId, withRoleIds);
        permissions.checkView(release);
        return release;
    }

    @Timed
    @Override
    public Release getRelease(String releaseId) {
        // keep old signature for scripts
        return getRelease(releaseId, false);
    }

    @Timed
    @Override
    public Release getArchivedRelease(String releaseId, boolean withRoleIds) {
        Release release = releaseService.findByIdInArchive(releaseId, withRoleIds);
        permissions.checkView(release);
        return release;
    }

    @Timed
    @Override
    public Release getArchivedRelease(String releaseId) {
        // keep old signature for scripts
        return getArchivedRelease(releaseId, false);
    }

    @Timed
    @Override
    public List<Task> getActiveTasks(String releaseId) {
        Release release = releaseService.findById(releaseId);
        permissions.checkView(release);

        return release.getActiveTasks();
    }

    @Timed
    @Override
    public Release start(String releaseId) {
        permissions.check(START_RELEASE, releaseId);
        checkArgument(!releaseService.isTemplate(releaseId), format("Could not start %s because it is a template.", releaseId));
        try {
            releaseService.checkCanBeStarted(releaseId);
            return releaseActorService.startRelease(releaseId, AUTHENTICATED_USER);
        } catch (IllegalStateException e) {
            logger.error("Could not start release " + releaseId, e);
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Timed
    @Override
    public Release updateRelease(String releaseId, Release release) {
        permissions.checkEdit(releaseId);
        return releaseActorService.updateRelease(releaseId, release);
    }

    @Timed
    @Override
    public Release updateRelease(Release release) {
        return updateRelease(release.getId(), release);
    }

    @Timed
    @Override
    public void delete(String releaseId) {
        permissions.check(ADMIN);
        releaseActorService.deleteRelease(releaseId);
    }

    @Timed
    @Override
    public Release abort(String releaseId, AbortRelease abortRelease) {
        permissions.check(ABORT_RELEASE, releaseId);
        return abort(releaseId, abortRelease.getAbortComment());
    }

    @Timed
    @Override
    public Release abort(String releaseId, String abortComment) {
        permissions.check(ABORT_RELEASE, releaseId);
        return releaseActorService.abortRelease(releaseId, abortComment);
    }

    @Timed
    @Override
    public List<Release> searchReleasesByTitle(String releaseTitle) {
        checkArgument(!isNullOrEmpty(releaseTitle), "Query parameter releaseTitle must be provided");

        return releaseSearchService.searchReleasesByTitle(releaseTitle, 0, Integer.MAX_VALUE, Integer.MAX_VALUE);
    }

    @Timed
    @Override
    public List<Variable> getVariables(String releaseId) {
        return variableComponent.getVariables(releaseId);
    }

    @Timed
    @Override
    public Map<String, String> getVariableValues(String releaseId) {
        List<Variable> variables = variableService.findGlobalVariablesOrEmpty().getVariables();
        List<Variable> folderVariables = folderVariableService.getAllFromAncestry(Ids.findFolderId(releaseId)).getVariables();
        variables.addAll(folderVariables);
        variables.addAll(variableComponent.getVariables(releaseId));
        return VariableHelper.getVariableValuesAsStrings(variables);
    }

    @Timed
    @Override
    public Variable getVariable(String variableId) {
        return variableComponent.getVariable(variableId);
    }

    @Timed
    @Override
    public Collection<Object> getVariablePossibleValues(String variableId) {
        return variableComponent.getVariablePossibleValues(variableId);
    }

    @Timed
    @Override
    public Boolean isVariableUsed(String variableId) {
        return variableComponent.isVariableUsed(variableId);
    }

    @Timed
    @Override
    public void replaceVariable(String variableId, VariableOrValue variableOrValue) {
        variableComponent.replaceVariable(variableId, variableOrValue);
    }

    @Timed
    @Override
    public void deleteVariable(String variableId) {
        variableComponent.deleteVariable(variableId);
    }

    @Timed
    @Override
    public Variable createVariable(String releaseId, com.xebialabs.xlrelease.api.v1.forms.Variable variable) {
        return variableComponent.createVariable(releaseId, variable);
    }

    @Timed
    @Override
    public List<Variable> updateVariables(String releaseId, List<Variable> variables) {
        return variableComponent.updateVariables(releaseId, variables);
    }

    @Timed
    @Override
    public Variable updateVariable(String variableId, Variable variable) {
        return variableComponent.updateVariable(variableId, variable);
    }

    @Timed
    @Override
    public Variable updateVariable(Variable variable) {
        return variableComponent.updateVariable(variable.getId(), variable);
    }

    @Timed
    @Override
    public List<String> getPermissions() {
        return XLReleasePermissions.getReleasePermissions();
    }

    @Timed
    @Override
    public List<TeamView> getTeams(String releaseId) {
        return teamFacade.getTeams(releaseId);
    }

    @Timed
    @Override
    public List<TeamView> setTeams(String releaseId, List<TeamView> teams) {
        return teamFacade.setTeams(releaseId, teams);
    }

    @Timed
    @Override
    public Release resume(String releaseId) {
        permissions.checkAny(releaseId, RESTART_PHASE, EDIT_RELEASE);
        try {
            return releaseActorService.resume(releaseId);
        } catch (IllegalStateException e) {
            logger.error("Could not resume release " + releaseId, e);
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Timed
    @Override
    public Release restartPhases(String releaseId, String phaseId, String taskId, PhaseVersion phaseVersion, boolean resumeRelease) {
        checkNotNull(releaseId, "releaseId");
        checkNotNull(phaseId, "fromPhaseId");
        checkNotNull(taskId, "fromTaskId");
        checkNotNull(phaseVersion, "phaseVersion");
        permissions.check(RESTART_PHASE, releaseId);
        try {
            return phaseService.restartPhase(releaseId, phaseId, taskId, phaseVersion, resumeRelease);
        } catch (IllegalStateException e) {
            logger.error("Could not restart phases of release " + releaseId, e);
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Override
    public Release restartPhase(Release release) {
        return restartPhase(release, false);
    }

    @Override
    public Release restartPhase(Release release, boolean resumeRelease) {
        checkNotNull(release, "given release");
        checkArgument(release.hasCurrentPhase(), "Release '%s' must have a current phase in order to restart phases", release.getTitle());
        checkArgument(!release.getCurrentPhase().getTasks().isEmpty(), "The given release does not have any task");

        return restartPhase(release, release.getCurrentPhase(), release.getCurrentPhase().getTask(0), PhaseVersion.ALL, resumeRelease);
    }

    @Override
    public Release restartPhase(Release release, Phase phase) {
        return restartPhase(release, phase, PhaseVersion.ALL);
    }

    @Override
    public Release restartPhase(Release release, Phase phase, PhaseVersion phaseVersion) {
        checkNotNull(release, "given release");
        checkNotNull(phase, "given phase");
        checkArgument(!phase.getTasks().isEmpty(), "The release with given phase does not have any task");

        return restartPhases(release.getId(), phase.getId(), phase.getTask(0).getId(), phaseVersion, false);
    }

    @Override
    public Release restartPhase(Release release, Phase phase, Task task) {
        return restartPhase(release, phase, task, PhaseVersion.ALL);
    }

    @Override
    public Release restartPhase(Release release, Phase phase, Task task, PhaseVersion phaseVersion) {
        checkNotNull(release, "given release");
        checkNotNull(phase, "given phase");
        checkNotNull(task, "given task");

        return restartPhases(release.getId(), phase.getId(), task.getId(), phaseVersion, false);
    }

    @Override
    public Release restartPhase(Release release, Phase phase, Task task, PhaseVersion phaseVersion, boolean resumeRelease) {
        checkNotNull(release, "given release");
        checkNotNull(phase, "given phase");
        checkNotNull(task, "given task");

        return restartPhases(release.getId(), phase.getId(), task.getId(), phaseVersion, resumeRelease);
    }
}
