package com.xebialabs.xlrelease.api.internal;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.sse.SseEventSink;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.util.StringUtils;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.core.rest.api.PermissionResource;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.events.EventBus;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.search.PlanItemSearchResult;
import com.xebialabs.xlrelease.search.ReleaseCountResults;
import com.xebialabs.xlrelease.search.ReleaseDateRangeResults;
import com.xebialabs.xlrelease.search.ReleaseSearchResult;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.views.*;
import com.xebialabs.xlrelease.views.converters.*;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.xlrelease.repository.IdType.DOMAIN;
import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isInFolder;
import static com.xebialabs.xlrelease.repository.Ids.isRoot;
import static com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.ABORT_RELEASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.CREATE_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_TEMPLATE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.RESTART_PHASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.START_RELEASE;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getReleasePermissions;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.getTemplatePermissions;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.Response.Status.OK;
import static javax.ws.rs.core.Response.status;
import static scala.jdk.javaapi.OptionConverters.toScala;

/**
 * Releases are the core business object manipulated in Digital.ai Release.
 * <p>
 * A release represents a number of activities in a certain time period, with people working on it.
 * </p>
 */
@Path("/releases")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class ReleaseResource {

    private ReleaseService releaseService;

    private PermissionChecker permissions;

    private ReleaseSearchService releasesSearch;

    private ReleaseActorService releaseActorService;

    private TaskAccessService taskAccessService;

    private ReleaseViewConverter releaseViewConverter;

    private PhaseViewConverter phaseViewConverter;

    private TasksViewConverter tasksViewConverter;

    private PlanItemViewConverter planItemViewConverter;

    private TeamMemberViewConverter teamMemberViewConverter;

    private TeamService teamService;

    private ReleaseFormConverter releaseFormConverter;

    private EventBus eventBus;

    private PhaseService phaseService;

    private CiSSEService ciSSEService;

    @Autowired
    public ReleaseResource(ReleaseService releaseService,
                           PermissionChecker permissions,
                           ReleaseSearchService releasesSearch,
                           ReleaseActorService releaseActorService,
                           TaskAccessService taskAccessService,
                           ReleaseViewConverter releaseViewConverter,
                           PhaseViewConverter phaseViewConverter,
                           TasksViewConverter tasksViewConverter,
                           PlanItemViewConverter planItemViewConverter,
                           TeamMemberViewConverter teamMemberViewConverter,
                           TeamService teamService,
                           ReleaseFormConverter releaseFormConverter,
                           EventBus eventBus,
                           PhaseService phaseService,
                           CiSSEService ciSSEService) {
        this.releaseService = releaseService;
        this.permissions = permissions;
        this.releasesSearch = releasesSearch;
        this.releaseActorService = releaseActorService;
        this.taskAccessService = taskAccessService;
        this.planItemViewConverter = planItemViewConverter;
        this.releaseViewConverter = releaseViewConverter;
        this.phaseViewConverter = phaseViewConverter;
        this.tasksViewConverter = tasksViewConverter;
        this.teamMemberViewConverter = teamMemberViewConverter;
        this.teamService = teamService;
        this.releaseFormConverter = releaseFormConverter;
        this.eventBus = eventBus;
        this.phaseService = phaseService;
        this.ciSSEService = ciSSEService;
    }

    /**
     * Returns the list of templates visible to the current user.
     *
     * @param page            next page to query
     * @param numberByPage    templates per page
     * @param templateFilters the search criteria
     * @return the list of templates
     */
    @POST
    @Timed
    @Path("templates/search")
    public TemplateOverviewResultsView searchTemplateOverview(@QueryParam("page") Long page,
                                                              @QueryParam("numberbypage") Long numberByPage,
                                                              TemplateFilters templateFilters) {

        TemplateOverviewResultsView result = releasesSearch.templatesOverview(
                templateFilters,
                (page != null) ? page : 0,
                ofNullable(numberByPage).orElse(releasesSearch.DEFAULT_NUMBER_OF_RELEASES_PER_PAGE())
        );
        return result;
    }

    /**
     * Returns the list of pairs (templateId, templateTitle) where the current user has given permission.
     *
     * @return the list of pairs (templateId, templateTitle)
     */
    @GET
    @Timed
    @Path("templates/search")
    public PlanItemSearchView getAllTemplateIds(@QueryParam("page") Long page,
                                                @QueryParam("folderId") String folderId,
                                                @QueryParam("numberbypage") Long numberByPage,
                                                @QueryParam("permission") String permissionName,
                                                @QueryParam("matchTemplate") String matchTemplate) {
        PlanItemSearchResult searchResult;
        if (!StringUtils.isEmpty(folderId) && !isRoot(folderId)) {
            permissions.checkViewFolder(folderId);
            permissions.canViewRelease(folderId);
        }
        if (permissionName == null) {
            searchResult = releasesSearch.findAllTemplateIdsAndTitles((page != null) ? page : 0, numberByPage, toScala(Optional.ofNullable(folderId)), toScala(Optional.ofNullable(matchTemplate)));
        } else {
            Permission permission = Permission.find(permissionName);
            if (permission == null) {
                throw new PermissionResource.UnknownPermissionException(permissionName);
            }
            searchResult = releasesSearch.findAllTemplateIdsAndTitles((page != null) ? page : 0, numberByPage, toScala(Optional.ofNullable(folderId)), permission, toScala(Optional.ofNullable(matchTemplate)));
        }
        return planItemViewConverter.toSearchView(searchResult);
    }

    /**
     * Returns the list of pairs (templateId, templateTitle) where the current user has given permission.
     *
     * @return the list of pairs (templateId, templateTitle) based on templateId
     */
    @GET
    @Timed
    @Path("template/search")
    public PlanItemSearchView getTemplateTitleById(@QueryParam("matchTemplate") String templateId) {
        PlanItemSearchResult templateResult;
        templateResult = releasesSearch.findTemplateTitleById(templateId);
        return planItemViewConverter.toSearchView(templateResult);
    }

    /**
     * Creates a template.
     *
     * @param releaseForm the information required to create the template
     * @return the created template
     */
    @POST
    @Timed
    @Path("templates")
    public ReleaseFullView createTemplate(ReleaseForm releaseForm) {
        if (releaseForm.getParentId() == null || ROOT_FOLDER_ID.equals(releaseForm.getParentId())) {
            permissions.check(CREATE_TEMPLATE);
        } else {
            permissions.check(EDIT_TEMPLATE, releaseForm.getParentId());
        }

        checkArgument(releaseForm.hasTitle(), "Template title is mandatory");
        checkArgument(releaseForm.hasValidDates(), "Dates are not valid");

        Release release;
        if (releaseForm.getTemplateId() == null) {
            release = releaseService.createTemplate(releaseFormConverter.toRelease(releaseForm), releaseForm.getParentId());
        } else {
            release = releaseService.copyTemplate(releaseForm.getTemplateId(), releaseForm.getTitle(), releaseForm.getDescription());
        }

        return getReleaseFullView(release);
    }

    /**
     * Deletes a template.
     *
     * @param templateId the identifier of the template
     * @return a status code 200 if the template was successfully removed
     */
    @DELETE
    @Timed
    @Path("templates/{templateId:.*Release[^/-]*}")
    public Response deleteTemplate(@PathParam("templateId") @IdParam String templateId) {
        if (isInFolder(templateId)) {
            permissions.check(EDIT_TEMPLATE, getParentId(templateId));
        } else {
            permissions.check(CREATE_TEMPLATE);
        }

        releaseActorService.deleteTemplate(templateId);
        return status(OK).build();
    }

    /**
     * Updates a template.
     *
     * @param templateId  the identifier of the template
     * @param releaseForm the changes to apply
     * @return the modified template
     */
    @PUT
    @Timed
    @Path("templates/{templateId:.*Release[^/-]*}")
    public ReleaseFullView updateTemplate(@PathParam("templateId") @IdParam String templateId, ReleaseForm releaseForm) {
        permissions.check(EDIT_TEMPLATE, templateId);
        return getReleaseFullView(releaseActorService.updateTemplate(templateId, releaseFormConverter.toRelease(releaseForm)));
    }

    /**
     * Searches releases.
     *
     * @param page            next page to query
     * @param numberByPage    releases per page
     * @param releasesFilters the search criteria
     * @return the list of matching releases
     */
    @POST
    @Timed
    @Path("search")
    public ReleaseOverviewSearchView searchReleases(@QueryParam("page") Long page,
                                                    @QueryParam("numberbypage") Long numberByPage,
                                                    @QueryParam("depth") Integer depth,
                                                    @QueryParam("properties") List<String> properties,
                                                    @QueryParam("extensions") List<String> extensions,
                                                    ReleasesFilters releasesFilters) {
        final int RELEASE_WITH_VARIABLES_AND_PHASES_DEPTH = 2;
        Integer loadDepth = ofNullable(depth).orElse(RELEASE_WITH_VARIABLES_AND_PHASES_DEPTH);
        boolean includeOriginTemplateData = extensions.contains("template");

        ReleaseSearchResult searchResult = releasesSearch.search(
                releasesFilters,
                (page != null) ? page : 0,
                ofNullable(numberByPage).orElse(releasesSearch.DEFAULT_NUMBER_OF_RELEASES_PER_PAGE()),
                loadDepth,
                includeOriginTemplateData);
        return releaseViewConverter.toOverviewSearchView(searchResult, properties, loadDepth, extensions);
    }

    /**
     * Count releases matching filter criteria.
     *
     * @param releasesFilters the search friteria
     * @return a map containing the number of releases (total) and the number by status
     */
    @POST
    @Timed
    @Path("count")
    public ReleaseCountResults countReleases(ReleasesFilters releasesFilters) {
        return releasesSearch.count(releasesFilters);
    }

    /**
     * Count releases matching filter criteria.
     *
     * @param releasesFilters the search friteria
     * @return a map containing the number of releases (total) and the number by status
     */
    @POST
    @Timed
    @Path("daterange")
    public ReleaseDateRangeResults getReleaseDateRange(ReleasesFilters releasesFilters) {
        return releasesSearch.getReleaseDateRange(releasesFilters);
    }

    /**
     * Creates a release.
     *
     * @param releaseForm the information required to create the release
     * @return the created release
     */
    @POST
    public ReleaseFullView createRelease(ReleaseForm releaseForm) {
        checkArgument(releaseForm.hasTitle(), "Release title is mandatory");
        if (!releaseForm.hasValidDates()) {
            throw new BadRequestException("Scheduled start date must be before due date");
        }

        Release release;
        if (releaseForm.isFromTemplate()) {
            String parentId = releaseForm.getParentId();
            String targetFolderId = parentId != null ? parentId : Ids.findFolderId(releaseForm.getTemplateId());
            permissions.checkIsAllowedToCreateReleaseFromTemplate(releaseForm.getTemplateId(), targetFolderId);
            release = releaseService.createFromTemplate(releaseForm.getTemplateId(), releaseFormConverter.toRelease(releaseForm), releaseForm.getParentId());
        } else {
            permissions.checkIsAllowedToCreateReleaseInFolder(releaseForm.getParentId());
            release = releaseService.createWithoutTemplate(releaseFormConverter.toRelease(releaseForm), releaseForm.getParentId());
        }

        return getReleaseFullView(release);
    }

    /**
     * Updates a template.
     *
     * @param releaseId   the release identifier
     * @param releaseForm the changes to apply
     * @return the updated release
     */
    @PUT
    @Timed
    @Path("{releaseId:((?!templates).)*Release[^/-]*}")
    public ReleaseFullView updateRelease(@PathParam("releaseId") @IdParam String releaseId, ReleaseForm releaseForm) {
        permissions.check(EDIT_RELEASE, releaseId);
        return getReleaseFullViewWithProperties(releaseActorService.updateRelease(releaseId, releaseFormConverter.toRelease(releaseForm)), singletonList(RISK_PROFILE));
    }

    /**
     * Returns the duration of a template
     *
     * @param templateId the template identifier
     * @return the template duration, in seconds.
     */
    @GET
    @Timed
    @Path("templates/{templateId:.*Release[^/-]*}/duration")
    public Integer getTemplateDuration(@PathParam("templateId") @IdParam String templateId) {
        Release template = releaseService.findByIdIncludingArchived(templateId);
        if (!template.isTemplate()) {
            throw new BadRequestException("templateId does not belong to a template");
        }
        permissions.checkView(template);
        Duration duration = releaseService.getDurationOf(template);
        return duration.toStandardSeconds().getSeconds();
    }

    /**
     * Starts a release.
     *
     * @param releaseId the release identifier
     * @return the release, which state will reflect the start
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/start")
    public ReleaseFullView startRelease(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.check(START_RELEASE, releaseId);

        releaseService.checkCanBeStarted(releaseId);

        return getReleaseFullView(releaseActorService.startRelease(releaseId, User.AUTHENTICATED_USER));
    }

    /**
     * Start releases with given identifiers.
     *
     * @param releaseIds list of release identifiers
     * @return list of successfully started release identifiers
     */
    @POST
    @Path("start")
    public BulkActionResultView startReleases(List<String> releaseIds) {
        List<String> allowedReleaseIds = permissions.filterStartableReleases(releaseIds);

        if (allowedReleaseIds.isEmpty()) {
            return new BulkActionResultView(allowedReleaseIds);
        }

        final List<Release> startedReleases = releaseActorService.startReleases(allowedReleaseIds, User.AUTHENTICATED_USER);

        final List<String> startedReleaseIds = startedReleases
                .stream()
                .map(Release::getId)
                .map(DOMAIN::convertToViewId)
                .collect(toList());

        return new BulkActionResultView(startedReleaseIds);
    }

    /**
     * Aborts a running release.
     *
     * @param releaseId the release identifier
     * @return the release, which state will reflect the operation
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/abort")
    public ReleaseFullView abortRelease(@PathParam("releaseId") @IdParam String releaseId, AbortReleaseForm abortReleaseForm) {
        permissions.check(ABORT_RELEASE, releaseId);
        return getReleaseFullView(releaseActorService.abortRelease(releaseId, abortReleaseForm.getAbortComment()));
    }

    /**
     * Aborts releases with given identifiers.
     *
     * @param abortReleaseForm parameters for aborting a release
     * @return list of successfully aborted release identifiers
     */
    @POST
    @Path("abort")
    public BulkActionResultView abortReleases(AbortReleaseForm abortReleaseForm) {
        List<String> allowedReleaseIds = permissions.filterAbortableReleases(abortReleaseForm.getReleasesIds());

        if (allowedReleaseIds.isEmpty()) {
            return new BulkActionResultView(allowedReleaseIds);
        }

        final List<Release> abortedReleases = releaseActorService.abortReleases(allowedReleaseIds, abortReleaseForm.getAbortComment());

        List<String> abortedReleaseIds = abortedReleases
                .stream()
                .map(Release::getId)
                .map(DOMAIN::convertToViewId).collect(toList());

        return new BulkActionResultView(abortedReleaseIds);
    }

    /**
     * Returns a release.
     *
     * @param releaseId the release identifier
     * @return the release
     */
    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}")
    public ReleaseFullView getRelease(@PathParam("releaseId") @IdParam String releaseId) {
        Release release = releaseService.findByIdIncludingArchived(releaseId);
        permissions.checkView(release);
        return getReleaseFullViewWithProperties(release, singletonList(RISK_PROFILE));
    }

    /**
     * Returns a status of release.
     *
     * @param releaseId the release identifier
     * @return the release status
     */
    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/status")
    public ReleaseStatus getReleaseStatus(@PathParam("releaseId") @IdParam String releaseId) {
        return releaseService.getStatus(releaseId);
    }

    /**
     * Moves a phase within a release.
     *
     * @param releaseId       the identifier of the release
     * @param movementIndexes the origin and target of the movement (described in terms of the parent container and the child index)
     * @return the moved phase
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/phases/move")
    public PhaseFullView movePhase(@PathParam("releaseId") @IdParam String releaseId, MovementIndexes movementIndexes) {
        permissions.checkEdit(releaseId);
        return getPhaseFullView(releaseActorService.movePhase(releaseId, movementIndexes));
    }


    /**
     * Moves a task within a release.
     *
     * @param releaseId       the identifier of the release
     * @param movementIndexes the origin and target of the movement (described in terms of the parent container and the child index)
     * @return the moved task
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/tasks/move")
    public TaskFullView moveTask(@PathParam("releaseId") @IdParam String releaseId, MovementIndexes movementIndexes) {
        permissions.checkEdit(releaseId);

        return getTaskFullView(releaseActorService.moveTask(releaseId, movementIndexes));
    }

    /**
     * Adds a new, empty phase to a release.
     *
     * @param releaseId the identifier of the release
     * @return the new task
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/phases/add")
    public PhaseFullView addPhase(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.checkEdit(releaseId);

        Phase phase = releaseActorService.addPhase(releaseId);
        return getPhaseFullView(phase);
    }

    /**
     * Duplicates a task in a release.
     *
     * @param releaseId    the identifier of the release
     * @param originTaskId the identifier of the task to duplicate
     * @return the duplicated task
     */
    @PUT
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/tasks/duplicate/{taskId:.*Task[^/-]*}")
    public TaskFullView duplicateTask(@PathParam("releaseId") @IdParam String releaseId, @PathParam("taskId") @IdParam String originTaskId) {
        permissions.checkEdit(releaseId);
        return getTaskFullView(releaseActorService.duplicateTask(releaseId, originTaskId));
    }

    /**
     * Duplicates a phase in a release.
     *
     * @param releaseId     the identifier of the release
     * @param originPhaseId the identifier of the phase to duplicate
     * @return the new phase
     */
    @PUT
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/phases/duplicate/{phaseId:.*Phase[^/-]*}")
    public PhaseFullView duplicatePhase(@PathParam("releaseId") @IdParam String releaseId, @PathParam("phaseId") @IdParam String originPhaseId) {
        permissions.checkEdit(releaseId);
        return getPhaseFullView(releaseActorService.duplicatePhase(releaseId, originPhaseId));
    }

    /**
     * Returns the permissions configured for a release.
     *
     * @param releaseId the identifier of the release
     * @return the permissions
     */
    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/permissions")
    public ReleasePermissionsView getPermissions(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.checkEditSecurity(releaseId);

        Release release = releaseService.findByIdIncludingArchived(releaseId);

        List<String> permissions = release.isTemplate() ? getTemplatePermissions() : getReleasePermissions();
        return new ReleasePermissionsView(permissions, release, teamMemberViewConverter);
    }

    /**
     * Restarts the release from a given phase. Which in effect copies phases between phaseId (from) and the current phase. Depending on the phaseVersion
     * it will have a different copy strategy.
     *
     * @param releaseId    the identifier of the release
     * @param phaseId      the identifier of the phase to restart from
     * @param taskId       the identifier of the task to restart from
     * @param phaseVersion the given phase version
     * @return the release
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/restartPhases")
    public ReleaseFullView restartPhases(@PathParam("releaseId") @IdParam String releaseId,
                                         @QueryParam("fromPhaseId") @IdParam String phaseId,
                                         @QueryParam("fromTaskId") @IdParam String taskId,
                                         @QueryParam("phaseVersion") PhaseVersion phaseVersion) {
        permissions.check(RESTART_PHASE, releaseId);
        return getReleaseFullView(phaseService.restartPhase(releaseId, phaseId, taskId, phaseVersion));
    }

    /**
     * Resume a release that had been paused as part of the restart operation.
     *
     * @param releaseId the identifier of the release
     * @return the release
     */
    @POST
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/resume")
    public ReleaseFullView resume(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.check(EDIT_RELEASE, releaseId);
        return getReleaseFullView(releaseActorService.resume(releaseId));
    }

    @DELETE
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/attachments/{attachmentId:.*Attachment[^/-]*}")
    public void deleteAttachment(@PathParam("releaseId") @IdParam String releaseId, @PathParam("attachmentId") @IdParam String attachmentId) {
        permissions.check(EDIT_RELEASE, releaseId);
        releaseActorService.deleteAttachment(releaseId, attachmentId);
    }

    @DELETE
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/{taskId:.*Task[^/-]*}/attachments/{attachmentId:.*Attachment[^/-]*}")
    public void deleteAttachmentFromTask(@PathParam("releaseId") @IdParam String releaseId,
                                         @PathParam("taskId") @IdParam String taskId,
                                         @PathParam("attachmentId") @IdParam String attachmentId) {
        permissions.checkIsAllowedToEditAttachmentsOnTask(taskId);
        releaseActorService.deleteAttachmentFromTask(releaseId, taskId, attachmentId);
    }

    /**
     * Return the list of assignable teams for the specified release.
     *
     * @param releaseId the identifier of the release
     * @return the list of assignable teams
     */
    @GET
    @Timed
    @Path("{releaseId:.*Release[^/-]*}/teams/assignable")
    public Collection<Team> getAssignableTeams(@PathParam("releaseId") @IdParam String releaseId) {
        permissions.checkReassignTaskPermission(releaseId);

        List<Team> teamsIncludingInherited = teamService.getEffectiveTeams(releaseId);
        Predicate<Team> withoutTemplateOwner = team -> !team.getTeamName().equals(Team.TEMPLATE_OWNER_TEAMNAME);
        return teamsIncludingInherited.stream().filter(withoutTemplateOwner).collect(toList());
    }

    @GET
    @Path("{releaseId:.*Release[^/-]*}/follow")
    @Produces(MediaType.SERVER_SENT_EVENTS)
    public void followRelease(@PathParam("releaseId") @IdParam String releaseId,
                           @Context SseEventSink sink) {
        Release release = releaseService.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS());
        permissions.checkView(release);
        ciSSEService.followCi(release.getId(), sink);
    }

    @GET
    @Path("tags")
    public Set<String> getAllTags() {
        return releaseService.getAllTags(500);
    }

    @GET
    @Path("tags/archived")
    public Set<String> getAllArchivedTags() {
        return releaseService.getAllArchivedTags(500);
    }

    private ReleaseFullView getReleaseFullView(final Release release) {
        return getReleaseFullViewWithProperties(release, emptyList());
    }

    private ReleaseFullView getReleaseFullViewWithProperties(final Release release, final List<String> properties) {
        return releaseViewConverter.toFullView(release, taskAccessService.getAllowedTaskTypesForAuthenticatedUser(), properties, singletonList("progress"), Integer.MAX_VALUE);
    }

    private PhaseFullView getPhaseFullView(final Phase phase) {
        return phaseViewConverter.toFullView(phase, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    private TaskFullView getTaskFullView(final Task task) {
        return tasksViewConverter.toFullView(task, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

}
