package com.xebialabs.xlrelease.api.internal;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.xlrelease.domain.Comment;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.repository.TaskRepository;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.CommentService;
import com.xebialabs.xlrelease.views.CommentView;
import com.xebialabs.xlrelease.views.converters.CommentViewConverter;

import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static java.lang.String.format;

/**
 * The comments of a task.
 */
@Path("/comments")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class CommentResource {

    private final CommentService commentService;

    private final TaskRepository taskRepository;

    private final PermissionChecker permissions;

    private final CommentViewConverter commentViewConverter;

    @Autowired
    public CommentResource(CommentService commentService, TaskRepository taskRepository, PermissionChecker permissions, CommentViewConverter commentViewConverter) {
        this.commentService = commentService;
        this.taskRepository = taskRepository;
        this.permissions = permissions;
        this.commentViewConverter = commentViewConverter;
    }

    @PUT
    @Timed
    @Path("{commentId}")
    public CommentView updateComment(@PathParam("commentId") @IdParam String commentId, CommentView commentView) {
        String taskId = getParentId(commentId);
        permissions.checkIsAllowedToCommentOnTask(taskId);

        String releaseId = releaseIdFrom(taskId);
        if (permissions.canEditTask(releaseId) || commentView.getAuthor().getUsername().equalsIgnoreCase(AUTHENTICATED_USER.getName())) {
            Task task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());
            Comment comment = commentService.update(task, commentId, commentView.getText());
            return commentViewConverter.toFullView(comment);
        } else {
            throw PermissionDeniedException.withMessage(format("You cannot comment on task [%s]", taskId));
        }
    }

    @DELETE
    @Timed
    @Path("{commentId}")
    public void deleteComment(@PathParam("commentId") @IdParam String commentId, CommentView commentView) {
        String taskId = getParentId(commentId);
        permissions.checkIsAllowedToCommentOnTask(taskId);

        String releaseId = releaseIdFrom(taskId);
        if (permissions.canEditTask(releaseId) || commentView.getAuthor().getUsername().equalsIgnoreCase(AUTHENTICATED_USER.getName())) {
            Task task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());
            commentService.delete(task, commentId);
        } else {
            throw PermissionDeniedException.withMessage(format("You cannot delete comment on task [%s]", taskId));
        }
    }
}
