package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.CustomScriptTask
import com.xebialabs.xlrelease.scheduler.workers.Worker.ExecuteJob
import com.xebialabs.xlrelease.scheduler.{CustomScriptTaskJob, NextCustomScriptTaskJob}
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.service.{CommentService, PhaseService}
import com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER
import org.springframework.stereotype.Component

@Component
class CustomScriptTaskWorker(val releaseActorService: ReleaseActorService,
                             phaseService: PhaseService,
                             val scriptServicesList: java.util.List[ScriptService],
                             commentService: CommentService
                            )
  extends Worker with ScriptServiceSupport with TaskWorkerFailureLogic {

  override def execute: ExecuteJob = {
    case CustomScriptTaskJob(taskRef) => doExecute(taskRef)
    case NextCustomScriptTaskJob(taskRef, _) => doExecute(taskRef)
  }

  private def doExecute(taskRef: TaskSoftReference[CustomScriptTask]) = {
    doRun(taskRef) {
      val task = taskRef.get()
      val r = scriptServiceFor(task).executeCustomScriptTask(task)
      handleCustomScriptTaskResult(r, task)
    }
  }

  def handleCustomScriptTaskResult(result: CustomScriptTaskResult, task: CustomScriptTask): Unit = {
    result match {
      case SuccessCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, results, authentication) =>
        withAuthentication(authentication) {
          val nextScriptPath = results.getNextScriptPath
          if (nextScriptPath != null) {
            releaseActorService.saveCustomScriptResults(taskId, results, executionId)
          } else {
            releaseActorService.finishCustomScriptTask(taskId, executionId, executionLog, attachmentId, Some(results))
          }
        }
      case FailureCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, authentication) =>
        withAuthentication(authentication) {
          releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, None)
        }
      case RestartCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, exception, authentication) =>
        withAuthentication(authentication) {
          releaseActorService.failScriptTask(taskId, null, executionId, None, None)
          try phaseService.restartPhase(exception.getReleaseId, exception.getPhaseId, exception.getTaskId, exception.getPhaseVersion, exception.isResumeRelease)
          catch {
            case e: Exception =>
              commentService.create(task, e.getMessage, AUTHENTICATED_USER, true);
          }
        }
    }
  }

}

