package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks._
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.calendar.{Blackout, SpecialDay}
import com.xebialabs.xlrelease.domain.events.{BlackoutCreatedEvent, BlackoutDeletedEvent, BlackoutUpdatedEvent}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.CalendarEntryRepository
import com.xebialabs.xlrelease.repository.Ids.{CALENDAR_ROOT, getName}
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.text.SimpleDateFormat
import java.util
import java.util.Date

@Service
class CalendarService @Autowired()(val calendarEntryRepository: CalendarEntryRepository,
                                   val ciIdService: CiIdService,
                                   val eventBus: EventBus) extends Logging {

  @Timed
  def getSpecialDay(specialDayId: String): SpecialDay = {
    if (!calendarEntryRepository.exists(specialDayId)) {
      val result = new SpecialDay
      result.setId(specialDayId)
      setDates(specialDayId, result)
      result
    } else {
      calendarEntryRepository.findById[SpecialDay](specialDayId)
    }
  }

  @Timed
  def setSpecialDay(specialDayId: String, specialDay: SpecialDay): SpecialDay = {
    if (specialDay.isBlank) {
      logger.debug(s"Removing special day [$specialDayId]")

      if (calendarEntryRepository.exists(specialDayId)) {
        calendarEntryRepository.delete(specialDayId)
      }
    } else {
      logger.debug(s"Saving special day [$specialDayId] with label [${specialDay.getLabel}] and color [${specialDay.getColor}]")
      specialDay.setId(specialDayId)
      setDates(specialDayId, specialDay)

      calendarEntryRepository.createOrUpdate(specialDay)
    }
    specialDay
  }

  @Timed
  def getSpecialDays(from: Date, to: Date): util.List[SpecialDay] = {
    calendarEntryRepository.findAllByTypeInRange[SpecialDay](Type.valueOf(classOf[SpecialDay]), from, to)
  }

  @Timed
  def addBlackout(blackout: Blackout): Blackout = addBlackout(CALENDAR_ROOT, blackout)

  @Timed
  def addBlackout(parentId: String, blackout: Blackout): Blackout = {
    logger.debug(s"Adding new blackout [${blackout.getLabel} for period [${blackout.getStartDate}] - [${blackout.getEndDate}] to parent [$parentId]")

    validate(blackout)
    blackout.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Blackout]), parentId))
    calendarEntryRepository.create(blackout)

    eventBus.publish(BlackoutCreatedEvent(blackout))

    blackout
  }

  @Timed
  def updateBlackout(updated: Blackout): Blackout = {
    logger.debug(s"Updating blackout [${updated.getId}]")
    validate(updated)

    val original = calendarEntryRepository.findById[Blackout](updated.getId)
    calendarEntryRepository.update(updated)

    eventBus.publish(BlackoutUpdatedEvent(original, updated))

    updated
  }

  @Timed
  def deleteBlackout(blackoutId: String): Unit = {
    logger.debug(s"Deleting blackout [$blackoutId]")

    val blackout = calendarEntryRepository.findById[Blackout](blackoutId)
    calendarEntryRepository.delete(blackoutId)

    eventBus.publish(BlackoutDeletedEvent(blackout))
  }

  @Timed
  def getBlackout(blackoutId: String): Blackout = calendarEntryRepository.findById(blackoutId)

  @Timed
  def isInBlackout(now: Date): Boolean = calendarEntryRepository.existsByTypeInRange(Type.valueOf(classOf[Blackout]), now, now)

  @Timed
  def getBlackouts(now: Date): util.List[Blackout] = getBlackouts(now, now)

  @Timed
  def getBlackouts(from: Date, to: Date): util.List[Blackout] =
    calendarEntryRepository.findAllByTypeInRange[Blackout](Type.valueOf(classOf[Blackout]), from, to)

  private def setDates(id: String, specialDay: SpecialDay) = {
    try {
      val dateFormat = new SimpleDateFormat(SpecialDay.DATE_FORMAT)
      dateFormat.setLenient(false)
      val specialDayDate = new DateTime(dateFormat.parse(getName(id)))
      specialDay.setStartDate(specialDayDate.withTimeAtStartOfDay.toDate)
      specialDay.setEndDate(specialDayDate.withHourOfDay(23).withMinuteOfHour(59).withSecondOfMinute(59).toDate)
    } catch {
      case e: Exception =>
        throw new IllegalArgumentException(s"Special day ID [$id] does not have a valid format - Configuration/Calendar/${SpecialDay.DATE_FORMAT}.", e)
    }
  }

  private def validate(blackout: Blackout): Unit = {
    val blackoutLabelCharLength = 1024
    checkNotNull(blackout.getLabel, "Blackout label")
    checkArgument(blackout.getLabel.length() <= blackoutLabelCharLength, "Blackout Label must be less than 1024 characters")
    checkNotNull(blackout.getStartDate, "Blackout start date")
    checkNotNull(blackout.getEndDate, "Blackout end date")
    checkArgument(blackout.getStartDate.before(blackout.getEndDate), "Blackout end date must be after start date.")
  }

}
