package com.xebialabs.xlrelease.api.v1.impl;

import com.codahale.metrics.annotation.Timed;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.TaskApi;
import com.xebialabs.xlrelease.api.v1.forms.Comment;
import com.xebialabs.xlrelease.api.v1.forms.Condition;
import com.xebialabs.xlrelease.api.v1.forms.StartTask;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.TaskRepository;
import com.xebialabs.xlrelease.repository.query.TaskBasicData;
import com.xebialabs.xlrelease.script.DefaultScriptService;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.TaskGranularPermissions;
import com.xebialabs.xlrelease.security.authentication.AuthenticationService;
import com.xebialabs.xlrelease.service.CommentService;
import com.xebialabs.xlrelease.service.GateConditionService;
import com.xebialabs.xlrelease.service.ReleaseService;
import com.xebialabs.xlrelease.service.UploadService;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.BadRequestException;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import static com.google.common.base.Objects.equal;
import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_USER_INPUT_TASK_VARIABLES_VALUES;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;
import static com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER;
import static com.xebialabs.xlrelease.variable.VariableHelper.containsVariables;


@Controller
public class TaskApiImpl implements TaskApi {
    private static final Logger logger = LoggerFactory.getLogger(TaskApiImpl.class);

    private final ReleaseActorService releaseActorService;
    private final CommentService commentService;
    private final GateConditionService gateConditionService;
    private final TaskRepository taskRepository;
    private final PermissionChecker permissions;
    private final TaskGranularPermissions taskPermissionChecker;
    private final ReleaseService releaseService;
    private final UploadService uploadService;
    private final AuthenticationService authenticationService;

    @Autowired
    public TaskApiImpl(ReleaseActorService releaseActorService,
                       TaskRepository taskRepository,
                       PermissionChecker permissions,
                       TaskGranularPermissions taskPermissionChecker,
                       GateConditionService gateConditionService,
                       UploadService uploadService,
                       ReleaseService releaseService,
                       AuthenticationService authenticationService,
                       CommentService commentService) {
        this.releaseActorService = releaseActorService;
        this.taskRepository = taskRepository;
        this.permissions = permissions;
        this.taskPermissionChecker = taskPermissionChecker;
        this.gateConditionService = gateConditionService;
        this.uploadService = uploadService;
        this.releaseService = releaseService;
        this.authenticationService = authenticationService;
        this.commentService = commentService;
    }

    @Timed
    @Override
    public List<Attachment> addAttachments(String taskId, HttpServletRequest request) throws IOException {
        permissions.checkIsAllowedToEditAttachmentsOnTask(taskId);

        ServletFileUpload upload = new ServletFileUpload();
        upload.setSizeMax(XlrConfig.getInstance().server().upload().maxSizeBytes());
        try {
            FileItemIterator fileItems = upload.getItemIterator(request);
            List<Attachment> attachments = this.uploadService.addAttachment(taskId, fileItems);

            if (attachments.isEmpty()) {
                logger.debug("No attachments uploaded caused by incorrect form body");
                throw new BadRequestException("Expected file upload");
            }

            return attachments;
        } catch (FileUploadException ex) {
            throw new BadRequestException(ex);
        }
    }

    @Timed
    @Override
    public Attachment addAttachment(String taskId, String fileName, byte[] fileByteArray) throws IOException {
        permissions.checkIsAllowedToEditAttachmentsOnTask(taskId);
        return this.uploadService.addAttachment(taskId, fileName, fileByteArray);
    }

    @Timed
    @Override
    public void deleteAttachment(String taskId, String attachmentId) {
        String releaseId = releaseIdFrom(taskId);
        permissions.checkIsAllowedToEditAttachmentsOnTask(taskId);
        releaseActorService.deleteAttachmentFromTask(releaseId, taskId, attachmentId);
    }

    @Timed
    @Override
    public Task getTask(String taskId) {
        permissions.checkView(releaseIdFrom(taskId));

        return taskRepository.findById(taskId);
    }

    @Timed
    @Override
    public Task copyTask(String taskId, String targetContainerId, int targetPosition) {
        final String releaseId = releaseIdFrom(taskId);
        permissions.checkCopyTask(releaseId);
        return releaseActorService.copyTask(taskId, targetContainerId, targetPosition);
    }

    @Timed
    @Override
    public Task updateTask(Task task) {
        return updateTask(task.getId(), task);
    }

    @Timed
    @Override
    public Task updateTask(String taskId, Task task) {
        return updateTask(taskId, task, false);
    }

    @Timed
    @Override
    public Task updateTask(String taskId, Task task, boolean overrideLock) {
        Set<TaskUpdateDirective> directives = taskPermissionChecker.getUpdateDirectives(releaseIdFrom(taskId));
        return releaseActorService.updateTask(taskId, task, directives, overrideLock);
    }

    @Timed
    @Override
    public Task changeTaskType(String taskId, String targetType) {
        checkArgument(!isNullOrEmpty(targetType), "Query parameter targetType must be provided");

        permissions.checkEditTask(releaseIdFrom(taskId));
        return releaseActorService.changeTaskType(taskId, Type.valueOf(targetType));
    }

    @Timed
    @Override
    public Task addTask(String containerId, Task task) {
        return addTask(containerId, task, null);
    }

    @Timed
    @Override
    public Task addTask(String containerId, Task task, Integer position) {
        String releaseId = releaseIdFrom(containerId);
        permissions.checkView(releaseId);
        permissions.checkEdit(releaseId);
        permissions.checkEditTask(releaseId);
        if (task.isLocked()) {
            permissions.checkLockTaskPermission(releaseId);
        }
        if (task.getPrecondition() != null) {
            permissions.checkEditPreconditionPermission(releaseId);
        }
        if (task.getFailureHandler() != null || task.isTaskFailureHandlerEnabled() || task.getTaskRecoverOp() != null) {
            permissions.checkEditFailureHandlerPermission(releaseId);
        }

        return releaseActorService.createTask(containerId, task, position);
    }

    @Timed
    @Override
    public Task completeTask(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when completing a task.");
        permissions.checkRelevantTaskTransitionPermission(taskId);
        checkArgument(areAllRequiredVariablesSet(taskId), "All required values must be set before completing the task");
        return releaseActorService.completeTask(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task completeTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when completing a task.");
        permissions.checkRelevantTaskTransitionPermission(taskId);
        checkArgument(areAllRequiredVariablesSet(taskId), "All required values must be set before completing the task");
        return releaseActorService.completeTask(taskId, comment);
    }

    private boolean areAllRequiredVariablesSet(String taskId) {
        final Task task = taskRepository.findById(taskId);
        return !isUserInputTask(task) || ((UserInputTask) task).getVariables().stream().noneMatch(v -> v.getRequiresValue() && v.isValueEmpty());
    }

    private boolean isUserInputTask(Task task) {
        return task instanceof UserInputTask;
    }

    @Timed
    @Override
    public Task skipTask(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when skipping a task.");
        permissions.checkRelevantTaskTransitionPermission(taskId);
        return releaseActorService.skipTask(taskId, comment.getComment(), AUTHENTICATED_USER);
    }

    @Timed
    @Override
    public Task skipTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when skipping a task.");
        permissions.checkRelevantTaskTransitionPermission(taskId);
        if (authenticationService.hasScriptTask() && authenticationService.hasVariablesHolderForScriptContext()) {
            DefaultScriptService.ScriptTaskResults scriptTaskResults = authenticationService.getCurrentVariablesHolderForScriptContext().createScriptTaskResults();
            releaseActorService.saveScriptResults(taskId, scriptTaskResults);
        }
        return releaseActorService.skipTask(taskId, comment, AUTHENTICATED_USER);
    }

    @Timed
    @Override
    public Task failTask(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when failing a task.");
        permissions.checkTaskTransitionPermission(taskId);
        return releaseActorService.failTaskManually(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task failTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when failing a task.");
        permissions.checkTaskTransitionPermission(taskId);
        return releaseActorService.failTaskManually(taskId, comment);
    }

    @Timed
    @Override
    public Task abortTask(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when aborting a task.");
        permissions.checkTaskTransitionPermission(taskId);
        checkArgument(!isNullOrEmpty(comment.getComment()), "Comment is mandatory when aborting a task.");
        return releaseActorService.abortTask(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task abortTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when aborting a task.");
        permissions.checkTaskTransitionPermission(taskId);
        return releaseActorService.abortTask(taskId, comment);
    }

    @Timed
    @Override
    public Task retryTask(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when retrying a task.");
        permissions.checkTaskTransitionPermission(taskId);
        checkArgument(!isNullOrEmpty(comment.getComment()), "Comment is mandatory when retrying a task.");
        return releaseActorService.retryTask(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task retryTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when retrying a task.");
        permissions.checkTaskTransitionPermission(taskId);
        return releaseActorService.retryTask(taskId, comment);
    }

    @Timed
    @Override
    public Task start(String taskId, Comment comment) {
        checkNotNull(comment, "Comment is mandatory when starting a pending task.");
        permissions.checkIsAllowedToStartTask(taskId);
        return releaseActorService.startTask(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task start(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when starting a pending task.");
        permissions.checkIsAllowedToStartTask(taskId);
        return releaseActorService.startTask(taskId, comment);
    }

    @Timed
    @Override
    public Task start(Task task, List<Variable> variables) {
        return start(task.getId(), new StartTask(variables));
    }

    @Timed
    @Override
    public Task start(String taskId, StartTask startTask) {
        permissions.checkIsAllowedToWorkOnTask(taskId);
        final List<Variable> taskVariables = startTask.getVariables();
        return releaseActorService.startTaskWithInput(taskId, taskVariables);
    }

    @Timed
    @Override
    public Task reopenTask(String taskId, Comment comment) {
        permissions.checkReopenTaskInRelease(releaseIdFrom(taskId));
        checkNotNull(comment, "Comment is mandatory when reopening a task.");
        checkArgument(!isNullOrEmpty(comment.getComment()), "Comment is mandatory when reopening a task.");
        return releaseActorService.reopenTask(taskId, comment.getComment());
    }

    @Timed
    @Override
    public Task reopenTask(String taskId, String comment) {
        checkArgument(!isNullOrEmpty(comment), "Comment is mandatory when reopening a task.");
        permissions.checkReopenTaskInRelease(releaseIdFrom(taskId));
        return releaseActorService.reopenTask(taskId, comment);
    }

    @Timed
    @Override
    public List<Variable> getVariables(String taskId) {
        final Task task = taskRepository.findById(taskId);
        permissions.checkViewTask(task);
        return task.getInputVariables();
    }

    @Override
    public List<Variable> updateInputVariables(String taskId, List<Variable> variables) {
        Release release = releaseService.findById(releaseIdFrom(taskId));
        Task task = release.getTask(taskId);
        checkArgument(isUserInputTask(task), "Can only add update variables to UserInputTask, not a " + task.getType().toString());
        UserInputTask userInputTask = (UserInputTask) task;
        for (Variable variable : variables) {
            checkArgument(userInputTask.getVariables().contains(variable), String.format("The variable [%s] does not belong to this task", variable.getKey()));
            permissions.checkEditVariable(release, userInputTask, variable);
            userInputTask.removeVariable(variable.getId());
            userInputTask.getVariables().add(variable);
        }
        UserInputTask updatedTask = (UserInputTask) releaseActorService.updateTask(taskId, userInputTask, new HashSet<>(Arrays.asList(UPDATE_USER_INPUT_TASK_VARIABLES_VALUES)));
        return updatedTask.getVariables();
    }

    @Timed
    @Override
    public Task commentTask(String taskId, Comment comment) {
        permissions.checkIsAllowedToCommentOnTask(taskId);
        Task task = taskRepository.findById(taskId);
        commentService.create(task, comment.getComment(), AUTHENTICATED_USER, true);
        return task;
    }

    @Timed
    @Override
    public Task commentTask(String taskId, String comment) {
        permissions.checkIsAllowedToCommentOnTask(taskId);
        Task task = taskRepository.findById(taskId);
        commentService.create(task, comment, AUTHENTICATED_USER, true);
        return task;
    }

    @Timed
    @Override
    public Task assignTask(String taskId, String username) {
        permissions.checkReassignTaskToUser(taskId, username);
        return releaseActorService.reassignTaskToOwner(taskId, username);
    }

    @Timed
    @Override
    public List<Task> searchTasksByTitle(String taskTitle, String phaseTitle, String releaseId) {
        checkArgument(!isNullOrEmpty(taskTitle), "Query parameter taskTitle must be provided");
        checkArgument(!isNullOrEmpty(releaseId), "Query parameter releaseId must be provided");
        Release release = releaseService.findById(releaseId);
        permissions.checkView(release);

        return release.getTasksByTitle(phaseTitle, taskTitle);
    }

    @Timed
    @Override
    public Task newTask() {
        return newTask("xlrelease.Task");
    }

    @Timed
    @Override
    public Task newTask(String type) {
        Task task = Task.fromType(type);
        task.setTitle("New task");
        return task;
    }

    @Override
    public Comment newComment(String commentText) {
        return new Comment(commentText);
    }

    @Timed
    @Override
    public void delete(String taskId) {
        permissions.checkEdit(releaseIdFrom(taskId));

        releaseActorService.deleteTask(taskId);
    }

    @Timed
    @Override
    public void deleteDependency(String dependencyId) {
        permissions.checkEditTask(releaseIdFrom(dependencyId));

        String taskId = Ids.getParentId(dependencyId);
        TaskBasicData taskBasicData = taskRepository.findTaskBasicData(taskId);
        checkArgument(isGateTask(taskBasicData), "Can only delete a dependency from a GateTask, not a " + taskBasicData.taskType().toString());

        releaseActorService.deleteDependency(taskId, dependencyId);
    }

    @Timed
    @Override
    public Dependency addDependency(String taskId, String targetId) {
        permissions.checkEditTask(releaseIdFrom(taskId));
        if (!containsVariables(targetId)) {
            permissions.checkView(releaseIdFrom(targetId));
        }

        TaskBasicData taskBasicData = taskRepository.findTaskBasicData(taskId);
        checkArgument(isGateTask(taskBasicData), "Can only add a dependency to a GateTask, not a " + taskBasicData.taskType().toString());

        return releaseActorService.createDependency(taskId, targetId);
    }

    @Timed
    @Override
    public GateCondition addCondition(String taskId, Condition condition) {
        permissions.checkEditTask(releaseIdFrom(taskId));

        TaskBasicData taskBasicData = taskRepository.findTaskBasicData(taskId);
        checkArgument(isGateTask(taskBasicData), "Can only add a condition to a GateTask, not a " + taskBasicData.taskType().toString());

        GateCondition gateCondition = releaseActorService.createGateCondition(taskId);
        gateCondition.setTitle(condition.getTitle());
        gateCondition.setChecked(condition.isChecked());

        return releaseActorService.updateGateCondition(gateCondition.getId(), gateCondition);
    }

    @Timed
    @Override
    public GateCondition updateCondition(String conditionId, Condition condition) {
        String gateId = Ids.getParentId(conditionId);
        permissions.checkIsAllowedToWorkOnTask(gateId);

        GateCondition previousCondition = gateConditionService.findById(conditionId);
        previousCondition.setChecked(condition.isChecked());
        if (!equal(previousCondition.getTitle(), condition.getTitle())) {
            permissions.checkEditTask(releaseIdFrom(conditionId));
            previousCondition.setTitle(condition.getTitle());
        }

        return releaseActorService.updateGateCondition(conditionId, previousCondition);
    }

    @Timed
    @Override
    public GateCondition updateCondition(GateCondition condition) {
        return releaseActorService.updateGateCondition(condition.getId(), condition);
    }

    @Timed
    @Override
    public void deleteCondition(String conditionId) {
        permissions.checkEditTask(releaseIdFrom(conditionId));

        releaseActorService.deleteGateCondition(conditionId);
    }

    @Timed
    @Override
    public void lockTask(final String taskId) {
        permissions.checkLockTaskPermission(releaseIdFrom(taskId));
        releaseActorService.lockTask(taskId);
    }

    @Timed
    @Override
    public void unlockTask(final String taskId) {
        permissions.checkLockTaskPermission(releaseIdFrom(taskId));
        releaseActorService.unlockTask(taskId);
    }

    private boolean isGateTask(TaskBasicData task) {
        return task.taskType().instanceOf(Type.valueOf("xlrelease.GateTask"));
    }
}
