package com.xebialabs.xlrelease.actors

import akka.actor._
import com.typesafe.akka.extension.quartz.QuartzSchedulerExtension
import com.xebialabs.xlplatform.cluster.NodeState.isActive
import com.xebialabs.xlrelease.actors.ArchivingActor.{JobDone, Tick}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.ArchivingScheduleService
import com.xebialabs.xlrelease.utils.QuartzUtils._

import scala.concurrent.Future
import scala.util.{Failure, Success}

object ArchivingActor {
  def props(service: ArchivingScheduleService, xlrConfig: XlrConfig) = Props(new ArchivingActor(service, xlrConfig))

  def name = "archiving"

  def scheduleName = "archivingSchedule"

  private[actors] case object Tick

  private case object JobDone

}

class ArchivingActor(service: ArchivingScheduleService, xlrConfig: XlrConfig) extends Actor with ActorLogging {

  private val scheduler = QuartzSchedulerExtension(context.system)
  private implicit val executionContext = context.system.dispatcher

  override def preStart() = {
    val cron = service.getArchivingJobCronSchedule.asQuartzCron
    scheduler.createSchedule(
      name = ArchivingActor.scheduleName,
      cronExpression = cron
    )
    log.info("Scheduling archiving job using cron " + cron)
    scheduler.schedule(ArchivingActor.scheduleName, self, Tick)
  }

  override def postStop(): Unit = {
    log.info("Stopping archiving job")
    scheduler.cancelJob(ArchivingActor.scheduleName)
    super.postStop()
  }

  override def receive: Actor.Receive = scheduleJob

  private def scheduleJob: Receive = {
    case Tick =>
      if (isActive) {
        log.debug("Starting archiving job")
        context.become(ignoreTicks)
        executeArchiveJob()
      }
  }

  private def ignoreTicks: Receive = {
    case Tick =>
      log.debug("Tick is ignored. Archive Job is in progress")
    case JobDone =>
      context.become(scheduleJob)
      log.debug("ArchivingActor ready");
  }

  private def executeArchiveJob(): Unit = {
    implicit val ec = xlrConfig.executors.auxiliaryExecutor.executionContext
    Future {
      service.processExpiredReleases()
    } onComplete {
      case Success(_) =>
        log.debug("Archiving job done")
        self ! JobDone
      case Failure(ex) =>
        log.debug("Error while calling archiving job")
        log.error(ex, ex.getMessage)
        self ! JobDone
    }
  }

}
