package com.xebialabs.xlrelease.upgrade.json

import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.json.FolderVariableUpgradeUtils._
import com.xebialabs.xlrelease.upgrade.{JsonUpgrade, UpgradeResult}
import com.xebialabs.xlrelease.variable.VariableHelper.withVariableSyntax
import org.codehaus.jettison.json.{JSONArray, JSONObject}
import org.springframework.stereotype.Component

import java.util.Collections
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

object FolderVariableUpgradeUtils {
  val FOLDER_VAR_PREFIX = "folder."
  val ACCEPTABLE_RELEASE_VAR_FOLDER_VAR_PREFIX = "folder_"

  def convertFolderVarKeyToReleaseVarKey(variableKey: String, alreadyUsed: collection.Set[String]): String = {
    val replacedKey = variableKey.replaceFirst(FOLDER_VAR_PREFIX, ACCEPTABLE_RELEASE_VAR_FOLDER_VAR_PREFIX)

    var variableIndex = 1
    var newKey = replacedKey
    while (alreadyUsed.contains(newKey)) {
      newKey = s"${replacedKey}_$variableIndex"
      variableIndex += 1
    }

    newKey
  }
}

@Component
class XLRelease860FolderVariableJsonUpgrade extends JsonUpgrade {
  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "8.6.0#4")

  /**
    * Modifies a release at JSON level
    *
    * @param release the release JSON object
    * @return { @link UpgradeResult}
    */
  //noinspection ScalaStyle
  override def performUpgrade(release: JSONObject): UpgradeResult = {
    val originalToReplacementKeys = mutable.Map.empty[String, String]

    def replaceArray(jsonArray: JSONArray): Unit = {
      for (i <- 0 until jsonArray.length()) {
        jsonArray.get(i) match {
          case _: String => originalToReplacementKeys.foreach { case (original, replacement) =>
            jsonArray.put(i, jsonArray.getString(i).replace(withVariableSyntax(original), withVariableSyntax(replacement)))
          }
          case obj: JSONObject => replaceObject(obj)
          case arr: JSONArray => replaceArray(arr)
          case _ => ()
        }
      }
    }

    def replaceObject(jsonObject: JSONObject): Unit = {
      jsonObject.keys().asScala.foreach {
        case key: String => jsonObject.get(key) match {
          case _: String => originalToReplacementKeys.foreach { case (original, replacement) =>
            jsonObject.put(key, jsonObject.getString(key).replace(withVariableSyntax(original), withVariableSyntax(replacement)))
          }
          case obj: JSONObject => replaceObject(obj)
          case arr: JSONArray => replaceArray(arr)
          case _ => ()
        }
      }
    }

    Option(release.optJSONArray("variables")).foreach { variableArray =>
      val variableKeysToIdxs = {
        for {
          i <- 0 until variableArray.length()
          obj <- Option(variableArray.optJSONObject(i))
          varKeyToIdx <- Try(obj.getString("key") -> i).toOption
        } yield varKeyToIdx
      }.toMap

      // Replace variable keys and store replacements
      val alreadyUsed = mutable.Set.empty[String] ++ variableKeysToIdxs.keys
      variableKeysToIdxs.foreach { case (variableKey, idx) =>
        val variableObject = variableArray.optJSONObject(idx)
        if (variableKey.startsWith(FOLDER_VAR_PREFIX)) {
          val newKey = convertFolderVarKeyToReleaseVarKey(variableKey, alreadyUsed)
          alreadyUsed += newKey
          originalToReplacementKeys += s"$variableKey" -> s"$newKey"
          variableObject.putOpt("key", newKey)
        }
      }
      // Replace trigger variable keys
      if (originalToReplacementKeys.nonEmpty) {
        for {
          triggerArray <- Option(release.optJSONArray("releaseTriggers"))
          i <- 0 until triggerArray.length()
          trigObj <- Option(triggerArray.optJSONObject(i))
          trigVarArray <- Option(trigObj.optJSONArray("variables"))
          j <- 0 until trigVarArray.length()
          variable <- Option(trigVarArray.optJSONObject(j))
          variableKey <- Try(variable.getString("key"))
          replacementKey <- originalToReplacementKeys.get(variableKey)
        } variable.putOpt("key", replacementKey)

        replaceObject(release)
      }
    }

    new UpgradeResult(originalToReplacementKeys.nonEmpty, Collections.emptyList())
  }
}
