package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.security.web.util.UrlUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import org.springframework.web.util.WebUtils;

import com.xebialabs.deployit.ServerConfiguration;

public class UrlRedirectRewriteFilter extends OncePerRequestFilter {

    private final boolean isFilterNotApplicable;
    private final String webContextRoot;

    public UrlRedirectRewriteFilter() {
        ServerConfiguration serverConfiguration = ServerConfiguration.getInstance();
        this.isFilterNotApplicable = serverConfiguration.getServerUrl().equals(normalisePathWithSlash(serverConfiguration.getDerivedServerUrl()));
        this.webContextRoot = serverConfiguration.getWebContextRoot();
    }

    @Override
    protected void doFilterInternal(final HttpServletRequest request, final HttpServletResponse response, final FilterChain filterChain) throws ServletException, IOException {
        filterChain.doFilter(request, RedirectResponseWrapper.wrapIfNecessary(response, webContextRoot));
    }

    private static String normalisePathWithSlash(String path) {
        return path.endsWith("/") ? path : path + "/";
    }


    @Override
    protected boolean shouldNotFilter(HttpServletRequest request) {
        return isFilterNotApplicable;
    }

    private static final class RedirectResponseWrapper extends HttpServletResponseWrapper {

        private String webContextRoot;

        private RedirectResponseWrapper(HttpServletResponse response, String webContextRoot) {
            super(response);
            this.webContextRoot = normalisePathWithSlash(webContextRoot);
        }

        @Override
        public void sendRedirect(String location) {
            setStatus(HttpStatus.SEE_OTHER.value());
            if (UrlUtils.isValidRedirectUrl(location)) {
                setHeader(HttpHeaders.LOCATION, location);
            } else {
                setHeader(HttpHeaders.LOCATION, webContextRoot + location);
            }
        }


        static HttpServletResponse wrapIfNecessary(HttpServletResponse response, String webContextRoot) {
            RedirectResponseWrapper wrapper =
                    WebUtils.getNativeResponse(response, RedirectResponseWrapper.class);
            return (wrapper != null ? response :
                    new RedirectResponseWrapper(response, webContextRoot));
        }
    }

}
