package com.xebialabs.xlrelease.events.handlers

import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.service.PendingPlanItemScheduledJobService
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date

@Service
@EventListener
class PendingTaskAutoStartEventHandler @Autowired()(val pendingPlanItemScheduledJobService: PendingPlanItemScheduledJobService)
  extends Logging {

  @Subscribe
  def onTaskCreatedOrTypeChangedEvent(event: TaskCreatedOrTypeChangedEvent): Unit = {
    val task: Task = event.task
    if (task.canScheduleTaskStart) {
      pendingPlanItemScheduledJobService.schedule(task)
    } else if (task.isPending && task.canStartNow) {
      // when blackout windows is deleted and task is pending, it should start immediately if scheduleStartDate is null
      task.setScheduledStartDate(new Date)
      pendingPlanItemScheduledJobService.schedule(task)
    } else if (!task.getRelease.isTemplate) {
      pendingPlanItemScheduledJobService.unschedule(task)
    }
  }

  @Subscribe
  def onTaskExecutionEvent(event: TaskExecutionEvent): Unit = {
    event match {
      case TaskStartedEvent(task) => pendingPlanItemScheduledJobService.unschedule(task)
      case TaskCompletedEvent(task, true) => pendingPlanItemScheduledJobService.unschedule(task)
      case TaskSkippedEvent(task, true) => pendingPlanItemScheduledJobService.unschedule(task)
      case TaskReopenedEvent(task) if task.canScheduleTaskStart => pendingPlanItemScheduledJobService.schedule(task)
      case TaskDelayedEvent(task) if task.canScheduleTaskStart => pendingPlanItemScheduledJobService.schedule(task)
      case _ => //do nothing
    }
  }

  @Subscribe
  def onTaskMovedEvent(event: TaskMovedEvent): Unit = {
    val task: Task = event.movedTask
    if (!task.getRelease.isTemplate) {
      pendingPlanItemScheduledJobService.unschedule(event.originalTaskId)
      if (task.canScheduleTaskStart) {
        pendingPlanItemScheduledJobService.schedule(task)
      }
    }
  }

  @Subscribe
  def onTaskDeletedEvent(event: TaskDeletedEvent): Unit = {
    if (!event.task.getRelease.isTemplate) {
      pendingPlanItemScheduledJobService.unschedule(event.task)
    }
  }

}
