package com.xebialabs.xlrelease.support.report.repository

import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.sql.SqlBuilder.{Db2Dialect, DerbyDialect, Dialect, OracleDialect}
import com.xebialabs.xlrelease.db.sql.{DatabaseInfo, Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.support.report._
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

object DataStatisticsRepository {
  val XL_VERSION_TABLE = "XL_VERSION"
  val XL_USERS_TABLE = "XL_USERS"
  val XL_ROLES_TABLE = "XL_ROLES"
  val XL_ROLE_ROLES_TABLE = "XL_ROLE_ROLES"
  val XL_ROLE_PRINCIPALS_TABLE = "XL_ROLE_PRINCIPALS"
  val XL_ROLE_PERMISSIONS_TABLE = "XL_ROLE_PERMISSIONS"
  val XLR_DASHBOARD_TABLE = "XLR_DASHBOARDS"
  val XLR_RELEASE_GROUPS_TABLE = "XLR_RELEASE_GROUPS"
  val XLR_DELIVERIES_TABLE = "XLR_DELIVERIES"
  val XLR_ACTIVITY_LOGS_TABLE = "XLR_ACTIVITY_LOGS"
  val DEPLOYMENTS_TABLE = "DEPLOYMENTS"
  val DEPLOYMENT_HISTORY_TABLE = "DEPLOYMENT_HISTORY"
  val XLR_TASK_RECORD_BUILD_TABLE = "XLR_TASK_RECORD_BUILD"
  val XLR_TASK_RECORD_PLAN_TABLE = "XLR_TASK_RECORD_PLAN"
  val TASK_RECORD_BUILD_TABLE = "TASK_RECORD_BUILD"
  val TASK_RECORD_PLAN_TABLE = "TASK_RECORD_PLAN"
  val XLR_TEMPLATE_REVISIONS_TABLE = "XLR_TEMPLATE_REVISIONS"
  val XLR_RISK_ASSESSMENTS_TABLE = "XLR_RISK_ASSESSMENTS"
  val XLR_ENVIRONMENTS_TABLE = "XLR_ENVIRONMENTS"
  val XLR_APPLICATIONS_TABLE = "XLR_APPLICATIONS"
  val TEST_QUERY = s"SELECT version FROM ${XL_VERSION_TABLE} WHERE component = 'xl-release'"
}

@Repository
//noinspection ScalaStyle
class DataStatisticsRepository @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val liveJdbcTemplate: JdbcTemplate,
                                            @Qualifier("xlrRepositorySqlDialect") val liveDialect: Dialect,
                                            @Qualifier("reportingJdbcTemplate") val archiveJdbcTemplate: JdbcTemplate,
                                            @Qualifier("reportingSqlDialect") val archiveDialect: Dialect) {

  import DataStatisticsRepository._

  def executeTestQuery: String =
    liveJdbcTemplate.queryForObject(TEST_QUERY, classOf[String])

  def countReleases(): Int = count(RELEASES.TABLE)

  def countReleasesByStatus(status: String): Int =
    count(RELEASES.TABLE, Sql(s"${RELEASES.STATUS} = ?", Seq(status)))

  def countReleasesInLastDays(days: Int): Int = {
    val fromDate = DateTime.now().minusDays(days).toDate
    val liveCount = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.STATUS} <> ?", Seq(fromDate, ReleaseStatus.TEMPLATE.value())))
    val archivedCount = count(REPORT_RELEASES_TABLE_NAME, Sql(s"$COMMON_START_DATE_COLUMN >= ? AND $REPORT_RELEASES_PRE_ARCHIVED = 0",
      Seq(fromDate)), archiveJdbcTemplate, archiveDialect)

    liveCount + archivedCount
  }

  def calculateCompletionRateLastMonth(): Float = {
    val startDate = DateTime.now().minusDays(60).withTimeAtStartOfDay().toDate
    val endDate = DateTime.now().minusDays(30).withTimeAtStartOfDay().toDate
    val startedReleases = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.START_DATE} <= ?  AND ${RELEASES.STATUS} <> (?)",
      Seq(startDate, endDate, ReleaseStatus.TEMPLATE.value())))
    val completedReleases = count(RELEASES.TABLE, Sql(s"${RELEASES.START_DATE} >= ? AND ${RELEASES.START_DATE} <= ?  AND ${RELEASES.STATUS} IN ('completed','aborted')",
      Seq(startDate, endDate)))
    avg(completedReleases, startedReleases)
  }

  def averageReleasesInLastDays(days: Int): Float = avg(countReleasesInLastDays(days), days)

  def averageReleasesPerMonthInLastYear(): Float = avg(countReleasesInLastDays(365), 12)

  def countReleasesWithDependencies(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(DISTINCT tasks.${TASKS.RELEASE_UID}) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${DEPENDENCIES.TABLE} dependencies ON dependencies.${DEPENDENCIES.GATE_TASK_UID} = tasks.${TASKS.CI_UID}")
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def findMaxDependenciesInRelease(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT tasks.${TASKS.RELEASE_UID}, COUNT(*) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${DEPENDENCIES.TABLE} dependencies ON dependencies.${DEPENDENCIES.GATE_TASK_UID} = tasks.${TASKS.CI_UID}")
    builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
    builder.withConditions(Seq(Sql(s"releases.${RELEASES.STATUS} != ?", Seq(ReleaseStatus.TEMPLATE.value()))), null)
    builder.groupBy(s"tasks.${TASKS.RELEASE_UID}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(1)
    findMax(builder)
  }

  def averageDependenciesPerRelease(): Float = avg(count(DEPENDENCIES.TABLE), count(RELEASES.TABLE))

  def countDependencies(): Int = count(DEPENDENCIES.TABLE)

  def countDistinctReleaseTags(): Int = countDistinct(TAGS.TABLE, TAGS.VALUE)

  def countTasks(): Int = count(TASKS.TABLE)

  def countDistinctTasksTags(): Int = countDistinct(TASK_TAGS.TABLE, TASK_TAGS.VALUE)

  def countTaskBackups(): Int = count(TASK_BACKUPS.TABLE)

  def countTaskComments(): Int = count(COMMENTS.TABLE)

  def countAttachments(): Int = count(ARTIFACTS.TABLE)

  def countAutomatedTasks(): Int =
    count(TASKS.TABLE, Sql(s"${TASKS.IS_AUTOMATED} = 1", Seq.empty))

  def countTasksByStatuses(statuses: Seq[TaskStatus]): Int =
    count(TASKS.TABLE, Sql(s"${TASKS.STATUS} IN (${statuses.map(_ => "?").mkString(", ")})", statuses.map(_.value())))


  def countPlannedTasks(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT COUNT(*) FROM ${TASKS.TABLE} tasks")
    builder.addJoin(s"INNER JOIN ${RELEASES.TABLE} releases ON releases.${RELEASES.CI_UID} = tasks.${TASKS.RELEASE_UID}")
    builder.withConditions(Seq(Sql(s"tasks.${TASKS.STATUS} IN (?) AND releases.${RELEASES.STATUS} <> ?",
      Seq(TaskStatus.PLANNED.value(), ReleaseStatus.TEMPLATE.value()))), "AND")
    val (sql, params) = builder.build()
    liveJdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  def averageTasksPerRelease(): Float = avg(count(TASKS.TABLE), countReleases())

  def countTopTaskTypeOccurrences(topTaskCount: Int): List[String] = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.TASK_TYPE}, COUNT(*) FROM ${TASKS.TABLE}")
    builder.groupBy(s"${TASKS.TASK_TYPE}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(topTaskCount)
    val (sql, params) = builder.build()
    liveJdbcTemplate.query(sql, countItemByOccurrenceMapper, params: _*).asScala.toList
  }

  def findMaxTasksInRelease(): Int = {
    val builder = new StatisticsSqlBuilder()(liveDialect)
    builder.select(s"SELECT ${TASKS.RELEASE_UID}, COUNT(*) FROM ${TASKS.TABLE}")
    builder.groupBy(s"${TASKS.RELEASE_UID}")
    builder.orderBy("COUNT(*) DESC")
    builder.limit(1)
    findMax(builder)
  }

  def countTemplates(): Int =
    count(RELEASES.TABLE, Sql(s"${RELEASES.STATUS} = ?", Seq(ReleaseStatus.TEMPLATE.value())))

  def countTriggers(): Int =
    count(TRIGGERS.TABLE)

  def countActiveTriggers(): Int =
    count(TRIGGERS.TABLE, Sql(s"${TRIGGERS.ENABLED} = 1", Seq.empty))

  def countFolders(): Int = count(FOLDERS.TABLE) - 1 // exclude the invisible root folder

  def maxFolderDepth(): Int =
    liveJdbcTemplate.queryForObject(s"SELECT MAX(${PATHS.DEPTH}) FROM ${PATHS.TABLE}", classOf[Integer])

  def countGlobalVariables(): Int =
    count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/variables/global%'", Seq.empty))

  def countFolderVariables(): Int = count(FOLDER_VARIABLES.TABLE)

  def countUsers(): Int = count(XL_USERS_TABLE)

  def countActiveUsersInLastDays(days: Int): Int =
    count(USER_PROFILE.TABLE, Sql(s"${USER_PROFILE.LAST_ACTIVE} >= ?", Seq(DateTime.now().minusDays(days).toDate)))

  def countGlobalRoles(): Int = count(XL_ROLES_TABLE, Sql("CI_ID = -1", Seq.empty))

  def countTeams(): Int = count(XL_ROLES_TABLE, Sql("CI_ID <> -1", Seq.empty))

  def countUserProfiles(): Int = count(USER_PROFILE.TABLE)

  def countEnabledUsers(): Int =
    count(USER_PROFILE.TABLE, Sql(s"${USER_PROFILE.ENABLED} = 1", Seq.empty))

  def countRoleRoles(): Int = count(XL_ROLE_ROLES_TABLE)

  def countRolePrincipals(): Int = count(XL_ROLE_PRINCIPALS_TABLE)

  def countRolePermissions(): Int = count(XL_ROLE_PERMISSIONS_TABLE)

  def countGlobalDashboards(): Int =
    count(s"${XLR_DASHBOARD_TABLE}", Sql("PARENT_ID IS NULL", Seq.empty))

  def countFolderDashboards(): Int =
    count(s"${XLR_DASHBOARD_TABLE}", Sql("PARENT_ID IS NOT NULL", Seq.empty))

  def countArchivedReleases(): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_PRE_ARCHIVED = 0", classOf[Integer])

  def countPreArchivedReleases(): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_PRE_ARCHIVED = 1", classOf[Integer])

  def oldestArchivedRelease(): String =
    archiveJdbcTemplate.queryForObject(s"SELECT MIN($COMMON_START_DATE_COLUMN) FROM $REPORT_RELEASES_TABLE_NAME", classOf[String])

  def countArchivedReleasesByState(status: String): Int =
    archiveJdbcTemplate.queryForObject(s"SELECT COUNT(*) FROM $REPORT_RELEASES_TABLE_NAME WHERE $COMMON_STATUS_COLUMN = '$status'", classOf[Integer])

  def countArchivedPhases(): Int = count(REPORT_PHASES_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedTasks(): Int = count(REPORT_TASKS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedAttachments(): Int = count(ARCHIVE_ATTACHMENTS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedTags(): Int = count(ARCHIVE_TAGS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedRoles(): Int = count(ARCHIVE_ROLEVIEWERS_TABLE_NAME, null, archiveJdbcTemplate)

  def countArchivedMembers(): Int = count(ARCHIVE_MEMBERVIEWERS_TABLE_NAME, null, archiveJdbcTemplate)

  def countReleaseGroups(): Int = count(XLR_RELEASE_GROUPS_TABLE)

  def countDeliveries(): Int = count(XLR_DELIVERIES_TABLE, Sql("STATUS <> 'template'", Seq.empty))

  def countDeliveryPatterns(): Int = count(XLR_DELIVERIES_TABLE, Sql("STATUS = 'template'", Seq.empty))

  def countActivityLogs(): Int = count(XLR_ACTIVITY_LOGS_TABLE)

  def countAttributes(): Int = count(FACETS.TABLE)

  def countDeploymentRecords(): Int = count(DEPLOYMENT_TASK_REPORTING_RECORD.TABLE)

  def countItsmRecords(): Int = count(ITSM_TASK_REPORTING_RECORD.TABLE)

  def countComplianceRecords(): Int = count(CODE_COMPLIANCE_TASK_REPORTING_RECORD.TABLE)

  def countBuildRecords(): Int = count(XLR_TASK_RECORD_BUILD_TABLE)

  def countPlanRecords(): Int = count(XLR_TASK_RECORD_PLAN_TABLE)

  def countDeployments(): Int = count(DEPLOYMENTS_TABLE, null, archiveJdbcTemplate)

  def countDeploymentHistories(): Int = count(DEPLOYMENT_HISTORY_TABLE, null, archiveJdbcTemplate)

  def countArchivedDeploymentRecords(): Int = count(Archive.DEPLOYMENT_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedItsmRecords(): Int = count(Archive.ITSM_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedComplianceRecords(): Int = count(Archive.CODE_COMPLIANCE_TASK_REPORTING_RECORD.TABLE, null, archiveJdbcTemplate)

  def countArchivedBuildRecords(): Int = count(TASK_RECORD_BUILD_TABLE, null, archiveJdbcTemplate)

  def countArchivedPlanRecords(): Int = count(TASK_RECORD_PLAN_TABLE, null, archiveJdbcTemplate)

  def countPermissionSnapshots(): Int = count(PERMISSION_SNAPSHOTS.TABLE)

  def countTemplateRevisions(): Int = count(XLR_TEMPLATE_REVISIONS_TABLE)

  def countRiskAssessments(): Int = count(XLR_RISK_ASSESSMENTS_TABLE)

  def countGlobalConfigurations(): Int = count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/Custom/%' AND ${CONFIGURATIONS.FOLDER_CI_UID} IS NULL", Seq.empty))

  def countFolderConfigurations(): Int = count(CONFIGURATIONS.TABLE, Sql(s"${CONFIGURATIONS.ID} LIKE 'Configuration/Custom/%' AND ${CONFIGURATIONS.FOLDER_CI_UID} IS NOT NULL", Seq.empty))

  def countApplications(): Int = count(XLR_APPLICATIONS_TABLE)

  def countEnvironments(): Int = count(XLR_ENVIRONMENTS_TABLE)

  def countCalendarEntries(): Int = count(CALENDAR_ENTRIES.TABLE)

  def findMainDbImplementation(): String = detectSqlDialect(liveJdbcTemplate)

  def findArchiveDbImplementation(): String = detectSqlDialect(archiveJdbcTemplate)

  def findAnalyticsIdentifier(): String = {
    val analyticsIdentifier = liveJdbcTemplate.query(s"SELECT ${METADATA.ENTRY} FROM ${METADATA.TABLE} WHERE ${METADATA.NAME} IN ('analytics_id')", new RowMapper[String] {
      override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(1)
    })
    if(analyticsIdentifier.isEmpty){
      null
    } else {
      analyticsIdentifier.get(0)
    }
  }

  def timeDbQueryNoHit(): Long = {
    var sql = ""
    liveDialect match {
      case OracleDialect(_) => sql = "SELECT CURRENT_TIMESTAMP FROM dual"
      case Db2Dialect(_) => sql = "SELECT CURRENT_TIMESTAMP FROM sysibm.sysdummy1"
      case DerbyDialect(_) => sql = "VALUES CURRENT_TIMESTAMP"
      case _ => sql = "SELECT CURRENT_TIMESTAMP"
    }
    val startTime = System.currentTimeMillis()
    liveJdbcTemplate.queryForObject(sql, classOf[String])
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  def timeDbQueryHit(): Long = {
    val startTime = System.currentTimeMillis()
    liveJdbcTemplate.queryForObject(TEST_QUERY, classOf[String])
    val endTime = System.currentTimeMillis()
    endTime - startTime
  }

  private def detectSqlDialect(jdbcTemplate: JdbcTemplate): String = {
    val dbInfo = DatabaseInfo(jdbcTemplate.getDataSource)
    dbInfo.toString
  }

  private def count(tableName: String, condition: Sql = null, jdbcTemplate: JdbcTemplate = liveJdbcTemplate, dialect: Dialect = liveDialect): Int = {
    val builder = new StatisticsSqlBuilder()(dialect)
    builder.select(s"SELECT COUNT(*) FROM ${tableName}")
    if (condition != null) {
      builder.withConditions(Seq(condition), "AND")
    }
    val (sql, params) = builder.build()
    jdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  private def countDistinct(tableName: String, columnName: String, jdbcTemplate: JdbcTemplate = liveJdbcTemplate, dialect: Dialect = liveDialect): Int = {
    val builder = new StatisticsSqlBuilder()(dialect)
    builder.select(s"SELECT COUNT(DISTINCT $columnName) FROM ${tableName}")
    val (sql, params) = builder.build()
    jdbcTemplate.queryForObject(sql, classOf[Integer], params: _*)
  }

  private def findMax(builder: StatisticsSqlBuilder) = {
    val (sql, params) = builder.build()
    val results = liveJdbcTemplate.query(sql, maxCountMapper, params: _*)
    if (results.size() > 0) {
      results.get(0)
    } else {
      0
    }
  }

  private val countItemByOccurrenceMapper: RowMapper[String] = (rs, _) =>
    rs.getString(1) + " - " + rs.getInt(2)

  private val maxCountMapper: RowMapper[Int] =
    (rs, _) =>
      rs.getInt(2)
}

class StatisticsSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[StatisticsSqlBuilder] {
  override def newInstance: StatisticsSqlBuilder = new StatisticsSqlBuilder
}

